/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: - Daniel Stucky (empolis GmbH) - initial API and implementation - Sebastian Voigt (brox IT Solutions
 * GmbH) - Juergen Schumacher (Attensity Europe GmbH) - changed for new job management.
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.impl;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.connectivity.ConnectivityException;
import org.eclipse.smila.connectivity.ConnectivityId;
import org.eclipse.smila.connectivity.deltaindexing.DeltaIndexingException;
import org.eclipse.smila.connectivity.deltaindexing.DeltaIndexingSessionException;
import org.eclipse.smila.connectivity.framework.Agent;
import org.eclipse.smila.connectivity.framework.AgentController;
import org.eclipse.smila.connectivity.framework.AgentCriticalException;
import org.eclipse.smila.connectivity.framework.AgentState;
import org.eclipse.smila.connectivity.framework.performancecounters.AgentPerformanceCounterHelper;
import org.eclipse.smila.connectivity.framework.performancecounters.ConnectivityPerformanceAgent;
import org.eclipse.smila.connectivity.framework.schema.config.DataConnectionID.DataConnectionType;
import org.eclipse.smila.connectivity.framework.schema.config.DataSourceConnectionConfig;
import org.eclipse.smila.connectivity.framework.schema.config.DeltaIndexingType;
import org.eclipse.smila.connectivity.framework.util.AgentControllerCallback;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.osgi.service.component.ComponentContext;

/**
 * Basic Implementation of a AgentController.
 */
public class AgentControllerImpl extends AbstractController implements AgentController {

  /** The Constant BUNDLE_ID. */
  private static final String BUNDLE_ID = "org.eclipse.smila.connectivity.framework";

  /** The LOG. */
  private final Log _log = LogFactory.getLog(getClass());

  /** A Map of active Agents. */
  private final java.util.Map<String, Agent> _activeAgents;

  /** A Map of AgentStates. */
  private final java.util.Map<String, AgentState> _agentStates;

  /** A Map of active CrawlThreads. (key: dataSourceId) to store performance counters for finished crawl threads */
  private final Map<String, AgentPerformanceCounterHelper<? extends ConnectivityPerformanceAgent>> _agentCounters;

  /** adapt agent controller methods to {@link AgentControllerCallback} interface by adding the job name. */
  private static class Callback implements AgentControllerCallback {

    /** agent controller to forward requests to. */
    private final AgentControllerImpl _controller;

    /** name of job to submit records to. */
    private final String _jobName;

    /** create instance. */
    public Callback(final AgentControllerImpl controller, final String jobName) {
      _controller = controller;
      _jobName = jobName;
    }

    /** {@inheritDoc} */
    @Override
    public boolean doDeltaIndexing(final DeltaIndexingType deltaIndexingType) {
      return _controller.doDeltaIndexing(deltaIndexingType);
    }

    /** {@inheritDoc} */
    @Override
    public boolean doCheckForUpdate(final DeltaIndexingType deltaIndexingType) {
      return _controller.doCheckForUpdate(deltaIndexingType);
    }

    /** {@inheritDoc} */
    @Override
    public boolean doDeltaDelete(final DeltaIndexingType deltaIndexingType) {
      return _controller.doDeltaDelete(deltaIndexingType);
    }

    /** {@inheritDoc} */
    @Override
    public void add(final String sessionId, final DeltaIndexingType deltaIndexingType, final Record record,
      final String hash) throws AgentCriticalException {
      _controller.add(sessionId, deltaIndexingType, record, hash, _jobName);
    }

    /** {@inheritDoc} */
    @Override
    public void delete(final String sessionId, final DeltaIndexingType deltaIndexingType, final ConnectivityId id)
      throws AgentCriticalException {
      _controller.delete(sessionId, deltaIndexingType, id, _jobName);
    }

    /** {@inheritDoc} */
    @Override
    public void unregister(final String sessionId, final DeltaIndexingType deltaIndexingType,
      final String dataSourceId) {
      _controller.unregister(sessionId, deltaIndexingType, dataSourceId);
    }
  }

  /**
   * Default Constructor.
   */
  public AgentControllerImpl() {
    if (_log.isTraceEnabled()) {
      _log.trace("Creating AgentControllerImpl");
    }
    _activeAgents = new HashMap<String, Agent>();
    _agentStates = new HashMap<String, AgentState>();
    _agentCounters = new HashMap<String, AgentPerformanceCounterHelper<? extends ConnectivityPerformanceAgent>>();
  }

  /**
   * @return CrawlerPerformanceCounterHelper for given data source id.
   */
  @Override
  public AgentPerformanceCounterHelper<? extends ConnectivityPerformanceAgent> getPerformanceCounterHelper(
    final String dataSourceId) {
    final Agent agent = _activeAgents.get(dataSourceId);
    if (agent != null) {
      return agent.getCounterHelper();
    }
    return _agentCounters.get(dataSourceId);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.AgentController#startAgent(String)
   */
  @Override
  public int startAgent(final String dataSourceId, final String jobName) throws ConnectivityException {
    // check parameters
    assertNotEmpty(dataSourceId, "dataSourceId");
    assertNotEmpty(jobName, "jobName");

    // check if data source is already used by another agent
    if (_activeAgents.containsKey(dataSourceId)) {
      throw new ConnectivityException("Can't start a new agent for DataSourceId '" + dataSourceId
        + "'. An agent is already started for it.");
    }

    try {
      final DataSourceConnectionConfig configuration = getConfiguration(BUNDLE_ID, dataSourceId);
      getConnectivityManager().checkJobIsActive(jobName);
      final Agent agent = createInstance(Agent.class, configuration.getDataConnectionID().getId());
      final int importRunId = agent.hashCode();

      // initialize the AgentState
      final AgentState agentState = new AgentState();
      agentState.setImportRunId(Integer.toString(importRunId));
      _agentStates.put(dataSourceId, agentState);

      String sessionId = null;
      if (doDeltaIndexing(configuration.getDeltaIndexing())) {
        sessionId = getDeltaIndexingManager().init(dataSourceId);
      }

      // start agent
      agent.start(new Callback(this, jobName), agentState, configuration, sessionId);
      agent.getCounterHelper().setJobName(jobName);
      _activeAgents.put(dataSourceId, agent);
      return importRunId;
    } catch (final ConnectivityException e) {
      throw e;
    } catch (final Exception e) {
      final String msg = "Error during start of agent using DataSourceId '" + dataSourceId + "'";
      _log.error(msg, e);
      throw new ConnectivityException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.AgentController#stopAgent(String)
   */
  @Override
  public void stopAgent(final String dataSourceId) throws ConnectivityException {
    // check parameters
    assertNotEmpty(dataSourceId, "dataSourceId");

    final Agent agent = _activeAgents.get(dataSourceId);
    if (agent == null) {
      final String msg =
        "Could not stop Agent for DataSourceId '" + dataSourceId + "'. No agent has been started for it.";
      _log.error(msg);
      throw new ConnectivityException(msg);
    }
    // stop agent
    try {
      agent.stop();
    } catch (final Exception e) {
      final String msg = "Error while stopping agent for DataSourceId '" + dataSourceId + "'";
      _log.error(msg, e);
      throw new ConnectivityException(msg, e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.AgentController#hasActiveAgents()
   */
  @Override
  public boolean hasActiveAgents() throws ConnectivityException {
    return !_activeAgents.isEmpty();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.AgentController#getAgentTasksState()
   */
  @Override
  public Map<String, AgentState> getAgentTasksState() {
    final HashMap<String, AgentState> states = new HashMap<String, AgentState>();
    states.putAll(_agentStates);
    return states;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.AgentController#getAvailableAgents()
   */
  @Override
  public Collection<String> getAvailableAgents() {
    return getAvailableFactories();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.AgentController#getAvailableConfigurations()
   */
  @Override
  public Collection<String> getAvailableConfigurations() {
    return getConfigurations(BUNDLE_ID, DataConnectionType.AGENT);
  }

  /**
   * Add the given record.
   * 
   * @param sessionId
   *          the delta indexing session Id
   * @param deltaIndexingType
   *          the DeltaIndexingType
   * @param record
   *          the record to add
   * @param hash
   *          the hash value used for delta indexing
   * @param jobName
   *          JobManager job to submit record to.
   */
  public void add(final String sessionId, final DeltaIndexingType deltaIndexingType, final Record record,
    final String hash, final String jobName) throws AgentCriticalException {
    if (record != null) {
      // set import run id as annotation on record
      ImportRunIdHelper.setImportRunIdAttribute(record, _agentStates.get(record.getSource()));

      try {
        boolean isUpdate = true;
        if (doCheckForUpdate(deltaIndexingType)) {
          isUpdate =
            getDeltaIndexingManager().checkForUpdate(sessionId,
              new ConnectivityId(record.getSource(), record.getId()), hash);
        }
        if (isUpdate) {
          // TODO: add compound management
          final boolean isCompound = false;

          // add record to connectivity manager
          getConnectivityManager().add(new Record[] { record }, jobName);

          // set delta indexing visited flag
          if (doDeltaIndexing(deltaIndexingType)) {
            getDeltaIndexingManager().visit(sessionId, new ConnectivityId(record.getSource(), record.getId()),
              hash, isCompound);
          }

          // execute delta delete for compounds only
          if (isCompound) {
            deleteDelta(sessionId, deltaIndexingType, record.getId());
          }

          // getPerformanceCounterHelper().incrementRecords();
        } // if
      } catch (final RuntimeException e) {
        // getPerformanceCounterHelper().addCriticalException(e);
        throw e;
      } catch (final ConnectivityException e) {
        // getPerformanceCounterHelper().addException(e);
        final String msg = "Error while deleting records for DataSourceId '" + record.getSource() + "'";
        if (e.isRecoverable()) {
          _log.error(msg, e);
        } else {
          throw new AgentCriticalException(msg, e);
        }
      } catch (final DeltaIndexingSessionException e) {
        // getPerformanceCounterHelper().addCriticalException(e);
        final String msg = "Error while deleting records for DataSourceId '" + record.getSource() + "'";
        _log.error(msg, e);
      } catch (final DeltaIndexingException e) {
        // getPerformanceCounterHelper().addException(e);
        final String msg = "Error while deleting records for DataSourceId '" + record.getSource() + "'";
        _log.error(msg, e);
      }

    }
  }

  /**
   * Delete the given id.
   * 
   * @param sessionId
   *          the delta indexing session Id
   * @param deltaIndexingType
   *          the DeltaIndexingType
   * @param id
   *          the id of the record to delete
   * @param jobName
   *          JobManager job to submit record to.
   */
  public void delete(final String sessionId, final DeltaIndexingType deltaIndexingType, final ConnectivityId id,
    final String jobName) throws AgentCriticalException {
    if (id != null) {
      final String dataSourceId = id.getDataSourceId();
      try {
        // TODO: add compound management
        final Record record = DataFactory.DEFAULT.createRecord();
        ImportRunIdHelper.setImportRunIdAttribute(record, _agentStates.get(record.getSource()));
        getConnectivityManager().delete(new Record[] { record }, jobName);

        // remove entry from delta indexing
        if (doDeltaDelete(deltaIndexingType)) {
          getDeltaIndexingManager().delete(sessionId, id);
        }

        // TODO: make sure delete also deletes elements of compounds, additional method in DIManager ?

      } catch (final RuntimeException e) {
        // getPerformanceCounterHelper().addCriticalException(e);
        throw e;
      } catch (final ConnectivityException e) {
        // getPerformanceCounterHelper().addException(e);
        final String msg = "Error while deleting records for DataSourceId '" + dataSourceId + "'";
        if (e.isRecoverable()) {
          _log.error(msg, e);
        } else {
          throw new AgentCriticalException(msg, e);
        }
      } catch (final DeltaIndexingSessionException e) {
        // getPerformanceCounterHelper().addCriticalException(e);
        final String msg = "Error while deleting records for DataSourceId '" + dataSourceId + "'";
        _log.error(msg, e);
      } catch (final DeltaIndexingException e) {
        // getPerformanceCounterHelper().addException(e);
        final String msg = "Error while deleting records for DataSourceId '" + dataSourceId + "'";
        _log.error(msg, e);
      }
    } // if
  }

  /**
   * {@inheritDoc}
   */
  public void unregister(final String sessionId, final DeltaIndexingType deltaIndexingType,
    final String dataSourceId) {
    final Agent removedAgent = _activeAgents.remove(dataSourceId);
    if (removedAgent != null) {
      _agentCounters.put(dataSourceId, removedAgent.getCounterHelper());
    }

    if (doDeltaIndexing(deltaIndexingType)) {
      try {
        getDeltaIndexingManager().finish(sessionId);
      } catch (final Exception e) {
        final String msg = "Error finishing delta indexing for DataSourceId '" + dataSourceId + "'";
        _log.error(msg, e);
      }
    }
  }

  /**
   * Deletes all elements of a compound id of a delta indexing run that were not visited.
   * 
   * @param sessionId
   *          the delta indexing session id
   * @param deltaIndexingType
   *          the DeltaIndexingType
   * @param compoundId
   *          the id of the compound record
   * @return the number of deleted Ids
   */
  private int deleteDelta(final String sessionId, final DeltaIndexingType deltaIndexingType, final String compoundId) {
    int count = 0;
    if (doDeltaDelete(deltaIndexingType)) {
      try {
        final Iterator<ConnectivityId> it = getDeltaIndexingManager().obsoleteIdIterator(sessionId, compoundId);
        if (it != null) {
          while (it.hasNext()) {
            final ConnectivityId id = it.next();
            if (id != null) {
              getDeltaIndexingManager().delete(sessionId, it.next());
              count++;
            }
          } // while
        } // if
      } catch (final Exception e) {
        final String msg = "Error during execution of deleteDelta for compoundId " + compoundId;
        _log.error(msg, e);
      }
    } // if
    return count;
  }

  /**
   * DS deactivate method.
   * 
   * @param context
   *          the ComponentContext
   * 
   * @throws Exception
   *           if any error occurs
   */
  protected void deactivate(final ComponentContext context) throws Exception {
    if (_log.isInfoEnabled()) {
      _log.info("Deactivating AgentController");
    }
    _lock.writeLock().lock();
    try {
      final Iterator<Map.Entry<String, Agent>> it = _activeAgents.entrySet().iterator();
      while (it.hasNext()) {
        final Map.Entry<String, Agent> entry = it.next();
        try {
          if (entry.getValue() != null) {
            entry.getValue().stop();
          }
        } catch (final Exception e) {
          _log.error("Error stopping Agent for data source " + entry.getKey(), e);
        }
      }
    } finally {
      _lock.writeLock().unlock();
    }
  }
}
