/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker;

import java.util.Map;

import org.eclipse.smila.objectstore.ObjectStoreService;

import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskworker.input.Inputs;
import org.eclipse.smila.taskworker.output.Outputs;

/**
 * The TaskContext interface provides methods for accessing input/output slot objects, accessing and logging to the
 * objectstore, and setting statistic counters.
 */
public interface TaskContext {

  /**
   * @return the current task
   */
  Task getTask();

  /**
   * @return Inputs object, for accessing objects of the input slots of the worker.
   */
  Inputs getInputs();

  /**
   * @return Outputs object, for accessing objects of the output slots of the worker.
   */
  Outputs getOutputs();

  /**
   * @return calculate and aggregate all counters in this task context and return the complete map. Call this
   *         immediately after the worker has finished.
   */
  Map<String, Number> getFinalCounters();

  /** @return a start time value to use in {@link #measureTime(String, long)} to measure a duration. */
  long getTimestamp();

  /**
   * measure time that has elapsed since the startTime value and add the duration to the named counter.
   * 
   * @param name
   *          the name of the counter
   * @param startTime
   *          the start time
   */
  void measureTime(String name, long startTime);

  /**
   * On each call, the value is added to the existing duration value for the given name. Values are assumed to describe
   * the duration in nanoseconds and are converted to a double duration value accordingly. In the final task counters,
   * these durations appear as duration.perform.xxx
   * 
   * @param name
   *          the name of the counter
   * @param duration
   *          the duration in nanoseconds to add.
   */
  void addDuration(String name, long duration);

  /**
   * On each call, the value is added to the existing duration value for the given name. Values are assumed to describe
   * a duration in seconds and are not changed. In the final task counters, these durations appear as
   * duration.perform.xxx
   * 
   * @param name
   *          the name of the counter
   * @param duration
   *          the duration in seconds to add.
   */
  void addDuration(String name, double duration);

  /**
   * On each call, the value is added to the existing counter value for the given name. No conversion (e.g. milliseconds
   * to seconds or sth. like that) is done. An example would be counting the numbers of objects in processing. The
   * counters will appear in the final task counters.
   * 
   * @param name
   *          the name of the counter
   * @param value
   *          the value to add to the counter
   */
  void addToCounter(String name, long value);

  /**
   * On each call, the value is added to the existing counter value for the given name. No conversion (e.g. milliseconds
   * to seconds or sth. like that) is done. An example would be counting the object sizes in processing. The counters
   * will appear in the final task counters.
   * 
   * @param name
   *          the name of the counter
   * 
   * @param value
   *          the value to add to the counter
   */
  void addToCounter(String name, double value);

  /**
   * @return <code>true</code> if current task is canceled, <code>false</code> otherwise.
   */
  boolean isCanceled();

  /**
   * cancel task execution.
   */
  void cancel();

  /**
   * @return the object store service used to access the object store.
   */
  ObjectStoreService getObjectStore();

  /** @return the task log, used for logging that can be accessed via REST/JSON API. */
  TaskLog getLog();
}
