/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr;

import org.apache.commons.lang.StringUtils;
import org.eclipse.smila.management.LocatedManagementAgentBase;

/**
 * The SolrManagerAgent class.
 * 
 * @author pwissel
 * 
 */
public class SolrServerManagerAgent extends LocatedManagementAgentBase {

  /**
   * The agent category.
   */
  private static final String CATEGORY = "Solr";

  /**
   * The agent name.
   */
  private static final String NAME = "ServerManager";

  /**
   * The SolrManager instance.
   */
  private final SolrServerManager _solrManager;

  /**
   * Default constructor.
   * 
   * @param solrManager
   *          the SolrManager instance.
   */
  public SolrServerManagerAgent(SolrServerManager solrManager) {
    _solrManager = solrManager;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.management.LocatedManagementAgentBase#getCategory()
   */
  @Override
  protected String getCategory() {
    return CATEGORY;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.management.LocatedManagementAgentBase#getName()
   */
  @Override
  protected String getName() {
    return NAME;
  }

  /**
   * Is embedded.
   * 
   * @return true if is EmbeddedSolrServer, false otherwise (HttpSolrServer).
   */
  public String isEmbedded() {
    return String.valueOf(_solrManager.getSolrProperties().isEmbedded());
  }

  /**
   * Get the solr.home path.
   * 
   * @return the solr.home path.
   */
  public String getSolrHome() {
    try {
      return _solrManager.getSolrHelper().getSolrHome().getPath();
    } catch (Exception exception) {
      return "Error invoking method: " + exception.getMessage();
    }
  }

  /**
   * Get the solr server URL, if solr is not embedded.
   * 
   * @return the solr server url.
   */
  public String getSolrServerURL() {
    return _solrManager.getSolrProperties().getServerUrl();
  }

  /**
   * Get all core names.
   * 
   * @return all core names.
   */
  public String getCoreNames() {
    return StringUtils.join(_solrManager.getCoreContainer().getCoreNames(), ";");
  }

  /**
   * Optimize.
   * 
   * @param coreName
   *          the core name.
   * @return the response message.
   */
  public String optimize(String coreName) {
    try {
      _solrManager.getSolrServer(coreName).optimize();
      return "Method successfully invoked.";
    } catch (Exception exception) {
      return "Error invoking method: " + exception.getMessage();
    }
  }

}
