/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.index;

import static java.lang.String.format;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.trimToNull;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrInputDocument;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.MissingParameterException;
import org.eclipse.smila.solr.Activator;
import org.eclipse.smila.solr.SolrConstants;
import org.eclipse.smila.solr.SolrConstants.AttributeOrAttachment;
import org.eclipse.smila.solr.SolrConstants.ExecutionMode;
import org.eclipse.smila.solr.SolrServerManager;
import org.eclipse.smila.solr.util.SolrUtils;

/**
 * SolrIndexPipelet class.
 * 
 * @author pwissel
 * 
 */
public class SolrIndexPipelet implements Pipelet {

  /**
   * The configuration error text.
   */
  private static final String CONF_ERROR = "Invalid Pipelet configuration -> ";

  /**
   * The configuration parameter missing error text.
   */
  private static final String CONF_ERROR_PARAMETER_MISSING = CONF_ERROR + "Parameter missing: ";

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(SolrIndexPipelet.class);

  /**
   * The configuration map.
   */
  private AnyMap _configuration;

  /**
   * The core fields sequence.
   */
  private AnySeq _coreFieldsSeq;

  /**
   * The execution mode.
   */
  private ExecutionMode _executionMode;

  /**
   * The solr server defined in the pipelet's BPEL config.
   */
  private SolrServer _defaultTargetCore;

  private String _defaultCoreName;

  /**
   * Constructor.
   */
  public SolrIndexPipelet() {
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.IPipelet#configure(org.eclipse.smila.processing.configuration.PipeletConfiguration)
   */
  @Override
  public void configure(AnyMap configuration) throws ProcessingException {
    if (_log.isDebugEnabled()) {
      _log.debug("Configure SolrIndexPipelet");
    }

    _configuration = configuration;
    try {
      if (StringUtils.isBlank(configuration.getStringValue(SolrConstants.EXECUTION_MODE))) {
        throw new MissingParameterException(CONF_ERROR_PARAMETER_MISSING + SolrConstants.EXECUTION_MODE);
      }
      _executionMode = ExecutionMode.valueOf(_configuration.getStringValue(SolrConstants.EXECUTION_MODE));

      switch (_executionMode) {
        case DELETE:
          break;
        case ADD:
        case UPDATE:
          _coreFieldsSeq = _configuration.getSeq(SolrConstants.CORE_FIELDS);
          for (Any coreFieldMap : _coreFieldsSeq) {
            if (!coreFieldMap.isMap()) {
              throw new ProcessingException("all items must be of type AnyMap in pipelet config: "
                + SolrConstants.CORE_FIELDS);
            }
          }
          break;
        default:
          throw new NotImplementedException("_executionMode: " + _executionMode);
      }

      // this does nothing except to issue a one time warning about the missing config.
      _defaultCoreName = trimToNull(_configuration.getStringValue(SolrConstants.CORE_NAME));
      if (isBlank(_defaultCoreName)) {
        _log.warn("there is no default core configured in the pipelet config! "
          + "Target core must be set now on each record dynamically via: " + SolrConstants.DYNAMIC_TARGET_CORE);
      }

    } catch (Exception exception) {
      throw new ProcessingException("Error while configure SolrIndexPipelet", exception);
    }

  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.Pipelet#process(org.eclipse.smila.blackboard.Blackboard,
   *      org.eclipse.smila.datamodel.id.String[])
   */
  @Override
  public String[] process(Blackboard blackboard, String[] recordIds) throws ProcessingException {
    if (recordIds != null) {
      if (_defaultCoreName != null) {
        try {
          _defaultTargetCore = getSolrManager().getSolrServer(_defaultCoreName);
        } catch (Exception e) {
          if (_log.isTraceEnabled()) {
            _log.warn(format("default core not reachable: %s. Exception: %s", _defaultCoreName, e.getMessage()));
          } // if
        }
      }

      switch (_executionMode) {
        case ADD:
          addRecords(blackboard, recordIds);
          break;
        case DELETE:
          delete(recordIds, _defaultTargetCore);
          break;
        default:
          throw new NotImplementedException("executionMode: " + _executionMode);
      }

    }
    return recordIds;
  }

  /**
   * Add to solr.
   * 
   * @param blackboard
   *          the blackboard.
   * @param recordIds
   *          the record ids.
   * @param server
   * @throws ProcessingException
   *           ProcessingException.
   */
  private void addRecords(Blackboard blackboard, String[] recordIds) throws ProcessingException {
    // since the target core can be set dynamically we must collect all docs per core and this is what this map does
    final Map<SolrServer, Collection<SolrInputDocument>> coreToDocsMap =
      new HashMap<SolrServer, Collection<SolrInputDocument>>();

    for (String id : recordIds) {
      traceRecord(blackboard, id);

      try {
        final SolrInputDocument doc = new SolrInputDocument();
        final AnyMap metadata = blackboard.getMetadata(id);

        // add id field
        doc.addField(SolrConstants.CORE_FIELD_ID, id);

        final SolrServer targetCore;
        if (metadata.containsKey(SolrConstants.DYNAMIC_TARGET_CORE)) {
          targetCore = getSolrManager().getSolrServer(metadata.getStringValue(SolrConstants.DYNAMIC_TARGET_CORE));
        } else {
          if (_defaultTargetCore == null) {
            throw new ProcessingException(
              "no dynamic core in record given while default core is null. check if the pipelet defines a valid default core or that the record carries the dynamicCore attribute");
          }

          targetCore = _defaultTargetCore;
        }
        Collection<SolrInputDocument> solrDocs = coreToDocsMap.get(targetCore);
        if (solrDocs == null) {
          solrDocs = new ArrayList<SolrInputDocument>(recordIds.length);
          coreToDocsMap.put(targetCore, solrDocs);
        }

        // set optional boost factor
        final Double boostFactor = metadata.getDoubleValue(SolrConstants.DOC_BOOST);
        if (boostFactor != null) {
          doc.setDocumentBoost(boostFactor.floatValue());
        }

        /* PERF: move these checks into config() | TM @ Jun 7, 2011 */
        /*
         * PERF: transform the map structure into a faster structure and do conversion and setting of default values
         * there | TM @ Jun 7, 2011
         */
        /*
         * BETTER: check that not target core field is mapped source field is mapped twice, as it will be overwridden.
         * since there are rare use cases for this (e.g. processing ensures that just on if the source field is actually
         * set), we just should issue a warning here) | TM @ Jun 7, 2011
         */
        for (Any coreFieldMap : _coreFieldsSeq) {
          final AnyMap fieldMap = coreFieldMap.asMap(); // configure validates this in advance

          final String fieldName = fieldMap.getStringValue(SolrConstants.CORE_FIELD_NAME);
          if (StringUtils.isBlank(fieldName)) {
            throw new MissingParameterException(CONF_ERROR_PARAMETER_MISSING + SolrConstants.CORE_FIELD_NAME);
          }
          String recSourceName = fieldMap.getStringValue(SolrConstants.SOURCE_NAME);
          if (StringUtils.isBlank(recSourceName)) {
            recSourceName = fieldName;
            if (_log.isTraceEnabled()) {
              _log.trace(format("core field mapping %s: no %s config'ed. Defaulting to field name", fieldName,
                SolrConstants.SOURCE_NAME));
            } // if
          }

          final String recSourceTypeString = fieldMap.getStringValue(SolrConstants.SOURCE_TYPE);
          final AttributeOrAttachment recSourceType;
          if (isBlank(recSourceTypeString)) {
            recSourceType = AttributeOrAttachment.ATTRIBUTE;
            if (_log.isTraceEnabled()) {
              _log.trace(format("core field mapping %s: no %s config'ed. Defaulting to %s", fieldName,
                SolrConstants.SOURCE_TYPE, recSourceType));
            }
          } else {
            recSourceType = AttributeOrAttachment.valueOf(recSourceTypeString.toUpperCase());
          }

          /* TODO | TM | add boost on field level | TM @ May 20, 2011 */
          switch (recSourceType) {
            case ATTRIBUTE: {
              final Any any = metadata.get(recSourceName);
              if (any != null) {
                if (any.isValue()) {
                  doc.addField(fieldName, any.asValue().getObject());
                } else if (any.isSeq()) {
                  final AnySeq asSeq = any.asSeq();
                  for (Any seqValues : asSeq) {
                    doc.addField(fieldName, seqValues.asValue().getObject());
                  }
                } else {
                  throw new ProcessingException(
                    "value type for indexing in solr not supported. Must be one of Value, Seq but is: "
                      + any.getValueType());
                }
              } else {
                if (_log.isTraceEnabled()) {
                  _log.trace("Record doesn't contain an attribute named: " + recSourceName);
                }
                break;
              }
            }
            case ATTACHMENT: {
              if (blackboard.hasAttachment(id, recSourceName)) {
                final byte[] value = blackboard.getAttachment(id, recSourceName);
                final String string = new String(value, "UTF-8");
                doc.addField(fieldName, string);
              } else {
                if (_log.isTraceEnabled()) {
                  _log.trace("Record doesn't have an attachment named: " + recSourceName);
                }
              }
              break;
            }
            default:
              throw new NotImplementedException("recSourceType: " + recSourceType);
          }

        }

        solrDocs.add(doc);
        if (_log.isInfoEnabled()) {
          _log.info("record added to document collection: " + id);

          if (_log.isTraceEnabled()) {
            _log.trace("solr document: " + doc.toString());
          } // if

        }
      } catch (Exception e) {
        final String msg = "Error while adding record with id: " + id;
        throw new ProcessingException(msg, e);
      } // try
    } // for

    try {
      for (Entry<SolrServer, Collection<SolrInputDocument>> entry : coreToDocsMap.entrySet()) {
        final SolrServer server = entry.getKey();
        final Collection<SolrInputDocument> docs = entry.getValue();
        final UpdateResponse solrResponse = server.add(docs);
        if (SolrUtils.responseStatusIsError(solrResponse)) {
          throw new ProcessingException("Error reported by solr reponse with status: " + solrResponse.getStatus());
        }

        // TODO: What is best way to commit?
        // _defaultTargetCore.commit(); // use autoCommit in solrconfig.xml
        if (_log.isDebugEnabled()) {
          _log.info(MessageFormat.format("document collection was added to solr in {1} ms, doc count = {0} ", //
            docs.size(), solrResponse.getElapsedTime()));
        }

      }
    } catch (Exception e) {
      throw new ProcessingException("Error while adding document collection to solr server.", e);
    }
  }

  /**
   * Log serialized record (trace level).
   * 
   * @param blackboard
   *          the blackboard.
   * @param id
   *          the id.
   */
  private void traceRecord(Blackboard blackboard, String id) {
    if (_log.isTraceEnabled()) {
      try {
        final Record record = blackboard.getRecord(id);
        final String serialize2string = XmlSerializationUtils.serialize2string(record).replace('\n', ' ');
        _log.trace("processing record: " + serialize2string);
      } catch (BlackboardAccessException e) {
        _log.trace("error on serializing the record for logging: ", e);
      }
    }
  }

  /**
   * Delete from solr.
   * 
   * @param recordIds
   *          the record ids.
   * @param server
   * @throws ProcessingException
   *           ProcessingException.
   */
  private void delete(String[] recordIds, SolrServer server) throws ProcessingException {
    for (String id : recordIds) {
      try {
        /*
         * NOTE: solr offers to delete a set of ids. but we deliberatly dont do this for logging purposes | tmenzel @
         * May 20, 2011
         */
        final UpdateResponse deleteResponse = server.deleteById(id);
        if (SolrUtils.responseStatusIsError(deleteResponse)) {
          throw new ProcessingException("Error reported by solr reponse while delete record with id: " + id);
        }
        if (_log.isDebugEnabled()) {
          final String msg =
            MessageFormat.format("Record deleted: Id: {0} Index: {1} time: {2}ms.", id,
              deleteResponse.getRequestUrl(), deleteResponse.getElapsedTime());
          _log.debug(msg);
        }
      } catch (Exception e) {
        throw new ProcessingException("Error while delete record with id: " + id, e);
      }
    }
  }

  /**
   * Get SolrManager.
   * 
   * @return the SolrManager.
   */
  public SolrServerManager getSolrManager() {
    /*
     * WORKAROUND: must be here instead of ctor/configure() due to init problems. suspicion: piplet tracker calling
     * configure() before the activator is called, despite it's lazy setting | TM @ Jun 7, 2011
     */

    return Activator.getInstance().getSolrManager();
  }

}
