/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.admin;

import static java.lang.String.format;

import java.util.Map;

import org.apache.solr.common.params.CoreAdminParams;
import org.apache.solr.common.params.CoreAdminParams.CoreAdminAction;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.core.SolrCore;
import org.apache.solr.handler.admin.CoreAdminHandler;
import org.apache.solr.request.LocalSolrQueryRequest;
import org.apache.solr.request.SolrQueryRequest;
import org.apache.solr.request.SolrQueryResponse;
import org.eclipse.smila.solr.util.SolrHelper;

/**
 * The SolrEmbeddedAdmin class.
 * 
 * @author pwissel
 * 
 */
public class SolrEmbeddedAdmin extends SolrBaseAdmin {

  /**
   * The CoreAdminHandler.
   */
  private final CoreAdminHandler _admin;

  /**
   * Constructor.
   * 
   * @param helper
   *          the SolrHelper.
   * @param container
   *          the CoreContainer.
   */
  public SolrEmbeddedAdmin(final SolrHelper helper, final CoreContainer container) {
    super(helper);
    _admin = new CoreAdminHandler(container);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.SolrAdminBase#alias(java.lang.String, java.lang.String)
   */
  @Override
  public String alias(final String coreName, final String aliasName) throws Exception {
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.ALIAS, coreName, aliasName);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.admin.SolrAdmin#create(java.lang.String)
   */
  @Override
  public String create(final String srcCore, String targetCore) throws Exception {
    _log.info(format("creating config by template: %s -> %s", srcCore, targetCore));
    _helper.copyCoreConfig(srcCore, targetCore);

    final String instanceDir = targetCore;
    final String dataDir = "data";
    return create(targetCore, instanceDir, dataDir);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.admin.SolrAdmin#create(java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  public String create(final String coreName, final String instanceDir, final String dataDir) throws Exception {
    final String[] name = { coreName };
    final String[] instanceDirParam = { instanceDir };
    final String[] dataDirParam = { dataDir };
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.CREATE, coreName, null);
    params.put(CoreAdminParams.NAME, name);
    params.put(CoreAdminParams.INSTANCE_DIR, instanceDirParam);
    params.put(CoreAdminParams.DATA_DIR, dataDirParam);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.SolrAdminBase#merge(java.lang.String, java.lang.String, java.lang.String)
   */
  @Override
  public String merge(final String coreName, final String indexDir1, final String indexDir2) throws Exception {
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.MERGEINDEXES, coreName, null);
    final String[] indexDir = { indexDir1, indexDir2 };
    params.put(CoreAdminParams.INDEX_DIR, indexDir);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.SolrAdminBase#reload(java.lang.String)
   */
  @Override
  public String reload(final String coreName) throws Exception {
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.RELOAD, coreName, null);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.SolrAdminBase#rename(java.lang.String, java.lang.String)
   */
  @Override
  public String rename(final String coreName, final String newName) throws Exception {
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.RENAME, coreName, newName);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.SolrAdminBase#status(java.lang.String)
   */
  @Override
  public String status(final String coreName) throws Exception {
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.STATUS, coreName, null);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.solr.SolrAdminBase#unload(java.lang.String)
   */
  @Override
  public String unload(final String coreName) throws Exception {
    final Map<String, String[]> params = buildAdminParams(CoreAdminAction.UNLOAD, coreName, null);
    final SolrCore core = _admin.getCoreContainer().getCore(coreName);
    final SolrQueryResponse response = handleRequest(core, params);
    return handleResponse(response);
  }

  /**
   * Handle request.
   * 
   * @param core
   *          the core.
   * @param params
   *          the params.
   * @return the query response.
   * @throws Exception
   *           Exception.
   */
  private SolrQueryResponse handleRequest(final SolrCore core, final Map<String, String[]> params) throws Exception {
    SolrQueryRequest req = null;
    SolrQueryResponse resp = null;

    // set class loader
    final ClassLoader cl = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(SolrQueryResponse.class.getClassLoader());

    try {
      req = new LocalSolrQueryRequest(core, params);

      if (_log.isDebugEnabled()) {
        _log.debug("Execute SolrQueryRequest with params: " + req.getParams());
      }

      // execute request.
      resp = new SolrQueryResponse();
      _admin.handleRequest(req, resp);

    } catch (Exception exception) {
      throw new Exception("Error while handle SolrQueryRequest", exception);
    } finally {
      // restore class loader
      Thread.currentThread().setContextClassLoader(cl);
    }
    return resp;
  }

  /**
   * Handle response.
   * 
   * @param response
   *          the response.
   * @return the response message.
   * @throws Exception
   *           Exception.
   */
  private String handleResponse(final SolrQueryResponse response) throws Exception {
    final Exception exception = response.getException();
    if (exception != null) {
      throw new Exception("Error while execute SolrQueryRequerst.", exception);
    }

    final StringBuilder output = new StringBuilder();
    final NamedList<?> header = response.getResponseHeader();
    if (header != null) {
      if (header.size() > 0) {
        output.append("ResponseHeader: ");
        output.append(header.toString() + "\n");
      }
    }

    final NamedList<?> log = response.getToLog();
    if (log != null) {
      if (log.size() > 0) {
        output.append("ToLog: ");
        output.append(log.toString() + "\n");
      }
    }

    final NamedList<?> values = response.getValues();
    if (values != null) {
      if (values.size() > 0) {
        output.append("Values: ");
        output.append(values.toString() + "\n");
      }
    }

    if (_log.isInfoEnabled()) {
      _log.info("SolrQueryRespoinse: " + output);
    }
    return output.toString();
  }
}
