package org.eclipse.smila.datamodel.ipc;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Map.Entry;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.datamodel.ValueFormatHelper;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.IpcStreamWriter;
import org.eclipse.smila.ipc.IpcToken;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.ipc.json.JsonFactory;

/**
 * Utility class for converting record objects to binary (BON) and json objects and vice versa. Note that the record Id
 * is NOT part of the BON representation by design.
 * 
 * @author stuc07
 */
public class IpcSerializationUtils {

  /** Encoding to use for String (de)serialization. */
  public static final String ENCODING = "UTF-8";

  /** attachment names for record (de)serialization. */
  private static final String ATTACHMENT_NAMES = "__attachmentNamesForIpc";

  /** BON IPC factory. */
  private final IpcFactory _binaryFactory = new BinaryFactory();

  /** JSON IPC factory. */
  private final IpcFactory _jsonFactory;

  /** helper for parsing date/datetimes. */
  private final ValueFormatHelper _formatHelper = new ValueFormatHelper();

  /** create instance with (JSON) pretty-printing enabled. */
  public IpcSerializationUtils() {
    this(true);
  }

  /** create instance with (JSON) printing as specified. */
  public IpcSerializationUtils(final boolean printPretty) {
    _jsonFactory = new JsonFactory(printPretty);
  }

  /** @return BON reader/writer factory. */
  public IpcFactory getBinaryFactory() {
    return _binaryFactory;
  }

  /**
   * Converts binary (BON) object to record object.
   * 
   * @param binaryObject
   *          input object
   * @return record object
   * @throws IOException
   *           in case of conversion error
   */
  public Record binaryObject2record(final byte[] binaryObject) throws IOException {
    ByteArrayInputStream bais = null;
    try {
      bais = new ByteArrayInputStream(binaryObject);
      return binaryStream2record(bais);
    } finally {
      IOUtils.closeQuietly(bais);
    }
  }

  /**
   * Converts binary (BON) stream to record object.
   * 
   * @param stream
   *          an input stream
   * @return a record with no Id set
   * @throws IOException
   *           in case of conversion error
   */
  public Record binaryStream2record(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _binaryFactory.newStreamReader(stream);
    return stream2record(reader);
  }

  /**
   * Converts binary (BON) object to record object.
   * 
   * @param binaryObject
   *          input object
   * @return record object
   * @throws IOException
   *           in case of conversion error
   */
  public Any binaryObject2any(final byte[] binaryObject) throws IOException {
    ByteArrayInputStream bais = null;
    try {
      bais = new ByteArrayInputStream(binaryObject);
      return binaryStream2any(bais);
    } finally {
      IOUtils.closeQuietly(bais);
    }
  }

  /**
   * Converts binary (BON) stream to record object.
   * 
   * @param stream
   *          an input stream
   * @return a record with no Id set
   * @throws IOException
   *           in case of conversion error
   */
  public Any binaryStream2any(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _binaryFactory.newStreamReader(stream);
    return stream2any(reader);
  }

  /**
   * Converts record object to binary (BON) object.
   * 
   * @param record
   *          input record
   * @return byte array representing the binary object.
   * @throws IOException
   *           in case of conversion error
   */
  public byte[] record2BinaryObject(final Record record) throws IOException {
    ByteArrayOutputStream baos = null;
    try {
      baos = new ByteArrayOutputStream();
      record2BinaryStream(baos, record);
      return baos.toByteArray();
    } finally {
      IOUtils.closeQuietly(baos);
    }
  }

  /**
   * Converts record object to binary (BON) stream.
   * 
   * @param stream
   *          output stream
   * @param record
   *          input record
   * @throws IOException
   *           in case of conversion error
   */
  public void record2BinaryStream(final OutputStream stream, final Record record) throws IOException {
    final IpcStreamWriter writer = _binaryFactory.newStreamWriter(stream);
    record2Stream(writer, record);
  }

  /**
   * Writes an Any object to a stream as BON.
   * 
   * @param stream
   *          output stream
   * @param object
   *          input object
   * @throws IOException
   *           in case of conversion error
   */
  public void any2BinaryStream(final OutputStream stream, final Any object) throws IOException {
    final IpcStreamWriter writer = _binaryFactory.newStreamWriter(stream);
    any2Stream(writer, object);
  }

  /**
   * Convert an Any object to a byte array.
   * 
   * @param object
   *          input object
   * @throws IOException
   *           in case of conversion error
   */
  public byte[] any2BinaryObject(final Any object) throws IOException {
    final ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    any2BinaryStream(outputStream, object);
    return outputStream.toByteArray();
  }

  /** @returns JSON reader/writer factory. */
  public IpcFactory getJsonFactory() {
    return _jsonFactory;
  }

  /**
   * Converts JSON string to record object.
   * 
   * @param jsonString
   *          input object
   * @return record object
   * @throws IOException
   *           in case of conversion error
   */
  public Record jsonObject2record(final String jsonString) throws IOException {
    ByteArrayInputStream bais = null;
    try {
      bais = new ByteArrayInputStream(jsonString.getBytes(ENCODING));
      final Record record = jsonStream2record(bais);
      return record;
    } finally {
      IOUtils.closeQuietly(bais);
    }
  }

  /**
   * Converts JSON stream to record object.
   * 
   * @param stream
   *          an input stream
   * @return a record with no Id set
   * @throws IOException
   *           in case of conversion error
   */
  public Record jsonStream2record(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _jsonFactory.newStreamReader(stream);
    try {
      return stream2record(reader);
    } finally {
      reader.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * Converts JSON stream to Any object.
   * 
   * @param stream
   *          an input stream
   * @return a record with no Id set
   * @throws IOException
   *           in case of conversion error
   */
  public Any jsonStream2any(final InputStream stream) throws IOException {
    final IpcStreamReader reader = _jsonFactory.newStreamReader(stream);
    try {
      return stream2any(reader);
    } finally {
      reader.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * Converts stream to record object.
   * 
   * @param reader
   *          an ipc reader (binary/json)
   * @return record parsed from stream, or null if end-of-stream reached.
   * @throws IOException
   *           IO errors from underlying stream.
   * @throws IllegalStateException
   *           parse errors.
   */
  public Record stream2record(final IpcStreamReader reader) throws IOException {
    IpcToken token = reader.nextToken();
    if (token == null) {
      return null;
    }
    if (token != IpcToken.OBJECT_START) {
      throw new IllegalStateException("Expected OBJECT_START Token; Token = " + token);
    }
    token = reader.nextToken();
    if (token != IpcToken.MAPPING_START) {
      throw new IllegalStateException("Expected MAPPING_START Token; Token = " + token);
    }
    final AnyMap metadata = readMap(reader);
    final Record record = DataFactory.DEFAULT.createRecord();
    if (metadata.containsKey(ATTACHMENT_NAMES)) {
      for (final Any attName : metadata.getSeq(ATTACHMENT_NAMES)) {
        record.setAttachment(attName.asValue().asString(), null);
      }
      metadata.remove(ATTACHMENT_NAMES);
    }
    record.getMetadata().putAll(metadata);
    token = reader.nextToken();
    if (token != IpcToken.OBJECT_END) {
      throw new IllegalStateException("Expected OBJECT_END Token; Token = " + token);
    }
    return record;
  }

  /**
   * Converts stream to record object.
   * 
   * @param reader
   *          an ipc reader (binary/json)
   * @return an Any, or null if end-of-stream reached.
   * @throws IOException
   *           in case of conversion error
   */
  public Any stream2any(final IpcStreamReader reader) throws IOException {
    IpcToken token = reader.nextToken();
    if (token == null) {
      return null;
    }
    if (token != IpcToken.OBJECT_START) {
      throw new IllegalStateException("Expected OBJECT_START Token; Token = " + token);
    }
    token = reader.nextToken();
    final Any any;
    if (token != IpcToken.OBJECT_END) {
      any = readAny(reader, token);
      token = reader.nextToken();
    } else {
      any = null;
    }
    if (token != IpcToken.OBJECT_END) {
      throw new IllegalStateException("Expected OBJECT_END Token; Token = " + token);
    }
    return any;
  }

  /**
   * Converts record object to JSON object.
   * 
   * @param record
   *          input record
   * @return JSON string representing the record.
   * @throws IOException
   *           in case of conversion error
   */
  public String record2JsonObject(final Record record) throws IOException {
    ByteArrayOutputStream baos = null;
    try {
      baos = new ByteArrayOutputStream();
      record2JsonStream(baos, record);
      return baos.toString(ENCODING);
    } finally {
      IOUtils.closeQuietly(baos);
    }
  }

  /**
   * Converts record object to JSON stream.
   * 
   * @param stream
   *          output stream
   * @param record
   *          input record
   * @throws IOException
   *           in case of conversion error
   */
  public void record2JsonStream(final OutputStream stream, final Record record) throws IOException {
    final IpcStreamWriter writer = _jsonFactory.newStreamWriter(stream);
    try {
      record2Stream(writer, record);
    } finally {
      writer.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * Converts AnyMap object to JSON stream.
   * 
   * @param stream
   *          output stream
   * @param object
   *          input record
   * @throws IOException
   *           in case of conversion error
   */
  public void map2JsonStream(final OutputStream stream, final AnyMap object) throws IOException {
    any2JsonStream(stream, object);
  }

  /**
   * Converts AnyMap object to JSON object.
   * 
   * @param object
   *          an AnyMap object
   * @return JSON string representing the input object.
   * @throws IOException
   *           in case of conversion error
   */
  public String map2JsonObject(final AnyMap object) throws IOException {
    ByteArrayOutputStream baos = null;
    try {
      baos = new ByteArrayOutputStream();
      map2JsonStream(baos, object);
      return baos.toString(ENCODING);
    } finally {
      IOUtils.closeQuietly(baos);
    }
  }

  /**
   * Converts Any object to JSON stream.
   * 
   * @param stream
   *          output stream
   * @param object
   *          input object
   * @throws IOException
   *           in case of conversion error
   */
  public void any2JsonStream(final OutputStream stream, final Any object) throws IOException {
    final IpcStreamWriter writer = _jsonFactory.newStreamWriter(stream);
    try {
      any2Stream(writer, object);
    } finally {
      writer.closeWithoutStream(); // do not close underlying stream
    }
  }

  /**
   * Converts Any object to IPC stream.
   * 
   * @param writer
   *          an IPC writer
   * @param object
   *          input object
   * @throws IOException
   *           in case of conversion error
   */
  public void any2Stream(final IpcStreamWriter writer, final Any object) throws IOException {
    writer.writeObjectStart();
    writeAny(writer, object);
    writer.writeObjectEnd();
  }

  /**
   * Converts record object to stream.
   * 
   * @param writer
   *          an ipc stream writer (bon/json)
   * @param record
   *          input record
   * @throws IOException
   *           in case of conversion error
   */
  public void record2Stream(final IpcStreamWriter writer, final Record record) throws IOException {
    // write out attachment names.
    final AnyMap recordMetadata = record.getMetadata();
    final Iterator<String> attNames = record.getAttachmentNames();
    while (attNames.hasNext()) {
      recordMetadata.add(ATTACHMENT_NAMES, recordMetadata.getFactory().createStringValue(attNames.next()));
    }
    any2Stream(writer, recordMetadata);
    recordMetadata.remove(ATTACHMENT_NAMES);
  }

  /**
   * read Map object from stream.
   * 
   * @param reader
   *          IPC stream
   * @return Map object
   * @throws IOException
   *           read error
   */
  private AnyMap readMap(final IpcStreamReader reader) throws IOException {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    IpcToken token;
    while ((token = reader.nextToken()) != IpcToken.MAPPING_END) {
      if (token != IpcToken.SCALAR_STRING) { // expect mapping key
        throw new IllegalStateException("Expected SCALAR_STRING Token; Token = " + token);
      }
      final String key = reader.currentStringValue();
      map.put(key, readAny(reader, reader.nextToken()));
    }
    return map;
  }

  /**
   * read a sequence object from the IPC stream. The reader must be positioned at the SEQUENCE_START token that starts
   * this sequence.
   * 
   * @param reader
   *          IPC reader
   * @return AnySeq object representing a sequence.
   * @throws IOException
   *           read error.
   */
  private AnySeq readSeq(final IpcStreamReader reader) throws IOException {
    final AnySeq anySeq = DataFactory.DEFAULT.createAnySeq();
    IpcToken token = null;
    while ((token = reader.nextToken()) != IpcToken.SEQUENCE_END) {
      anySeq.add(readAny(reader, token));
    }
    return anySeq;
  }

  /**
   * Reads a single value.
   * 
   * @param reader
   *          the IpcStreamReader
   * @param token
   *          the current IpcToken
   * @return a Value object
   * @throws IOException
   *           if any error occurs
   */
  private Value readValue(final IpcStreamReader reader, final IpcToken token) throws IOException {
    Value value = null;
    switch (token) {
      case SCALAR_BOOL:
        value = DataFactory.DEFAULT.createBooleanValue(reader.currentBoolValue());
        break;
      case SCALAR_DOUBLE:
        value = DataFactory.DEFAULT.createDoubleValue(reader.currentDoubleValue());
        break;
      case SCALAR_INT:
        value = DataFactory.DEFAULT.createLongValue(reader.currentLongValue());
        break;
      case SCALAR_STRING:
      default:
        // can also be DATE and DATETIME, let the data factory guess
        value = DataFactory.DEFAULT.tryDateTimestampParsingFromString(reader.currentStringValue());
    }
    return value;
  }

  /**
   * read Any object from Ipc stream.
   * 
   * @param reader
   *          reader
   * @param token
   *          current token
   * @return parsed Any
   * @throws IOException
   *           if any error occurs.
   */
  private Any readAny(final IpcStreamReader reader, final IpcToken token) throws IOException {
    Any any = null;
    switch (token) {
      case MAPPING_START:
        any = readMap(reader);
        break;
      case SEQUENCE_START:
        any = readSeq(reader);
        break;
      default:
        any = readValue(reader, token);
        break;
    }
    return any;
  }

  /**
   * write AnySeq object to the IPC stream.
   * 
   * @param writer
   *          IPC stream, the attribute name has already been written.
   * @param seq
   *          the sequence to write
   * @throws IOException
   *           write error
   */
  private void writeSeq(final IpcStreamWriter writer, final AnySeq seq) throws IOException {
    writer.writeSequenceStart();
    final Iterator<Any> seqElements = seq.iterator();
    while (seqElements.hasNext()) {
      final Any any = seqElements.next();
      writeAny(writer, any);
    }
    writer.writeSequenceEnd();
  }

  /**
   * write AnyMap object to the IPC stream.
   * 
   * @param writer
   *          IPC stream
   * @param object
   *          AnyMap object to write.
   * @throws IOException
   *           write error
   */
  private void writeMap(final IpcStreamWriter writer, final AnyMap object) throws IOException {
    if (object != null) {
      writer.writeMappingStart();
      if (!object.isEmpty()) {
        for (final Entry<String, Any> childEntry : object.entrySet()) {
          final String childKey = childEntry.getKey();
          writer.writeMappingKey(childKey);
          final Any any = childEntry.getValue();
          writeAny(writer, any);
        }
      }
      writer.writeMappingEnd();
    }
  }

  /**
   * Writes a Value to the IPC stream.
   * 
   * @param writer
   *          the IpcStreamWriter
   * @param value
   *          the value to write
   * @throws IOException
   *           if any error occurs
   */
  private void writeValue(final IpcStreamWriter writer, final Value value) throws IOException {
    if (value != null) {
      if (value.isBoolean()) {
        writer.writeScalarBoolean(value.asBoolean());
      } else if (value.isLong()) {
        writer.writeScalarLong(value.asLong());
      } else if (value.isDouble()) {
        writer.writeScalarDouble(value.asDouble());
      } else {
        // DATE and DATETIME are not directly supported, so String is used for them, too.
        writer.writeScalarString(value.asString());
      } // switch
    } // if
  }

  /**
   * Writes a object to the IPC stream.
   * 
   * @param writer
   *          the IpcStreamWriter
   * @param object
   *          the object to write
   * @throws IOException
   *           if any error occurs
   */
  private void writeAny(final IpcStreamWriter writer, final Any object) throws IOException {
    if (object != null) {
      if (object.isMap()) {
        writeMap(writer, (AnyMap) object);
      } else if (object.isSeq()) {
        writeSeq(writer, (AnySeq) object);
      } else if (object.isValue()) {
        writeValue(writer, (Value) object);
      }
    }
  }

}
