/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Ivan Churkin (brox IT Solutions GmbH) - initial creator Alexander Eliseyev (brox IT Solutions GmbH)
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.performancecounters;

import java.util.Date;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.management.MeasureUnit;

/**
 * The Class CrawlerPerformanceAgent.
 */
public class ConnectivityPerformanceAgent extends ConnectivityPerformanceAgentBase {

  /** Bytes per KByte. */
  private static final int BYTES_PER_KBYTE = 1024;

  /** The _start date. */
  private Date _startDate;

  /** The _end date. */
  private Date _endDate;

  /** The import run id. */
  private String _importRunId;

  /** The job name. */
  private String _jobName;

  /**
   * Instantiates a new crawler performance agent.
   */
  public ConnectivityPerformanceAgent() {
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.performancecounters
   *      .CrawlerControllerPerformanceAgent#getAverageAttachmentTransferRate()
   */
  @MeasureUnit("KBytes/Second")
  public double getAverageAttachmentTransferRate() {
    return (getAttachmentBytesTransfered().getNextSampleValue() / getLastRunTime()) * MILLISECS_IN_SEC
      / BYTES_PER_KBYTE;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.performancecounters
   *      .CrawlerControllerPerformanceAgent#getOverallAverageDeltaIndicesProcessingTime()
   */
  @MeasureUnit("ms")
  public double getOverallAverageDeltaIndicesProcessingTime() {
    return (getLastRunTime() / getDeltaIndices().getNextSampleValue());
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.performancecounters
   *      .CrawlerControllerPerformanceAgent#getOverallAverageRecordsProcessingTime()
   */
  @MeasureUnit("ms")
  public double getOverallAverageRecordsProcessingTime() {
    return (getLastRunTime() / getRecords().getNextSampleValue());
  }

  /**
   * Gets the last run time.
   * 
   * @return the last run time
   */
  private long getLastRunTime() {
    if (_startDate == null) {
      return 0;
    }
    if (_startDate != null && _endDate == null) {
      return (System.currentTimeMillis() - _startDate.getTime());
    }
    return _endDate.getTime() - _startDate.getTime();
  }

  /**
   * Gets the start date.
   * 
   * @return the start date
   */
  public Date getStartDate() {
    return _startDate;
  }

  /**
   * Sets the start date.
   * 
   * @param date
   *          the new start date
   */
  // TODO: make @Hidden
  public void setStartDate(final Date date) {
    _startDate = date;
  }

  /**
   * Gets the end date.
   * 
   * @return the end date
   */
  public Date getEndDate() {
    return _endDate;
  }

  /**
   * Sets the job name.
   * 
   * @param jobName
   *          the new job name
   */
  // TODO: make @Hidden
  public void setJobName(final String jobName) {
    _jobName = jobName;
  }

  /**
   * Gets the job name.
   * 
   * @return the job name
   */
  public String getJobName() {
    return _jobName;
  }

  /**
   * Sets the end date.
   * 
   * @param endDate
   *          the new end date
   */
  // TODO: make @Hidden
  public void setEndDate(final Date endDate) {
    _endDate = endDate;
  }

  /**
   * @return the import run id.
   */
  public String getImportRunId() {
    return _importRunId;
  }

  /**
   * @param importRunId
   *          the importRunId
   */
  // TODO: make @Hidden
  public void setImportRunId(final String importRunId) {
    _importRunId = importRunId;
  }

  /**
   * @return AnyMap representation of the counters.
   */
  public AnyMap toAny() {
    final AnyMap any = DataFactory.DEFAULT.createAnyMap();
    any.put("jobName", getJobName());
    any.put("attachmentBytesTransfered", getAttachmentBytesTransfered().getNextSampleValue());
    any.put("attachmentTransferRate", getAttachmentTransferRate().getNextSampleValue());
    any.put("averageAttachmentTransferRate", getAverageAttachmentTransferRate());
    any.put("averageDeltaIndicesProcessingTime", getAverageDeltaIndicesProcessingTime().getNextSampleValue());
    any.put("averageRecordsProcessingTime", getAverageRecordsProcessingTime().getNextSampleValue());
    any.put("deltaIndices", getDeltaIndices().getNextSampleValue());
    if (getEndDate() != null) {
      any.put("endDate", DataFactory.DEFAULT.createDateTimeValue(getEndDate()));
    }
    any.put("errorBuffer", String.valueOf(getErrorBuffer().getErrors()));
    any.put("exceptions", getExceptions().getNextSampleValue());
    any.put("exceptionsCritical", getExceptionsCritical().getNextSampleValue());
    any.put("importRunId", getImportRunId());
    any.put("overallAverageDeltaIndicesProcessingTime", getOverallAverageDeltaIndicesProcessingTime());
    any.put("overallAverageRecordsProcessingTime", getOverallAverageRecordsProcessingTime());
    any.put("records", getRecords().getNextSampleValue());
    if (getStartDate() != null) {
      any.put("startDate", DataFactory.DEFAULT.createDateTimeValue(getStartDate()));
    }
    return any;
  }
}
