package org.eclipse.smila.jobmanager;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.jobmanager.internal.AccessAny;

/**
 * parsed form of parameter definitions.
 * 
 */
public class ParameterDefinition extends DefinitionBase {
  /** external name of type "any". */
  public static final String TYPENAME_ANY = "any";

  /** property for type of parameters. */
  public static final String KEY_TYPE = "type";

  /** property for optional parameters. */
  public static final String KEY_OPTIONAL = "optional";

  /** parameter property: list of accepted values. */
  public static final String KEY_VALUES = "values";

  /** parameter property: multiple values allowed? */
  public static final String KEY_MULTI = "multi";

  /** parameter property: list of entries of map parameters. */
  public static final String KEY_ENTRIES = "entries";

  /** parameter property keys to keep when creating a description without details. */
  public static final String[] MAIN_PARAMETER_KEYS = { KEY_NAME, KEY_TYPE, KEY_OPTIONAL, KEY_MULTI, KEY_VALUES };

  /** the expression denoting a variable parameter name. */
  public static final String VARIABLE_NAME_EXPRESSION = "<[^>]*>";

  /** the parameter's type. null for type "any". */
  private final ValueType _type;

  /** flag for optional parameters. */
  private final boolean _isOptional;

  /** flag for multi-value parameters. */
  private final boolean _isMulti;

  /** entries of map parameters. */
  private List<ParameterDefinition> _entries;

  /** allowed values for scalar parameters. */
  private List<Value> _values;

  /** constructor to parse a definition from an Any. */
  public ParameterDefinition(final AnyMap definitionAny) throws InvalidConfigException {
    super(definitionAny);
    _type = parseType(definitionAny);
    _isOptional = AccessAny.getBoolean(definitionAny, KEY_OPTIONAL, false);
    _isMulti = AccessAny.getBoolean(definitionAny, KEY_MULTI, false);
    if (definitionAny.containsKey(KEY_ENTRIES)) {
      if (_type != ValueType.MAP) {
        throw new InvalidConfigException("Parameter " + getName() + " of type " + getTypeName()
          + " cannot specify entries.");
      }
      _entries = parseParameters(AccessAny.getSeq(definitionAny, KEY_ENTRIES));
    }
    if (definitionAny.containsKey(KEY_VALUES)) {
      if (_type == null || _type == ValueType.MAP) {
        throw new InvalidConfigException("Parameter " + getName() + " of type " + getTypeName()
          + " cannot spefify values.");
      }
      final AnySeq valueAnys = AccessAny.getSeq(definitionAny, KEY_VALUES);
      if (valueAnys.isEmpty()) {
        throw new InvalidConfigException("Value list for parameter " + getName() + " must not be emtpy.");
      }
      _values = new ArrayList<Value>();
      for (final Any value : valueAnys) {
        if (!value.isValue() || value.getValueType() != _type) {
          throw new InvalidConfigException("Value " + value + " for parameter " + getName() + " is not of type "
            + getTypeName() + ", but " + value.getValueType());
        }
        _values.add(value.asValue());
      }
    }
  }

  /**
   * parse type from parameter definition.
   */
  private ValueType parseType(final AnyMap definitionAny) throws InvalidConfigException {
    if (definitionAny.containsKey(KEY_TYPE)) {
      final String typeString = AccessAny.getString(definitionAny, KEY_TYPE);
      if (TYPENAME_ANY.equalsIgnoreCase(typeString)) {
        return null;
      }
      final ValueType type;
      try {
        type = ValueType.valueOf(typeString.toUpperCase());
      } catch (final Exception ex) {
        throw new InvalidConfigException("Invalid parameter type " + typeString + " for parameter " + getName(), ex);
      }
      if (type == ValueType.SEQ) {
        throw new InvalidConfigException("Type 'seq' for parameter " + getName()
          + " is not a valid parameter type.");
      }
      return type;
    } else {
      return ValueType.STRING;
    }
  }

  /**
   * @param parameters
   *          parameters with additional attributes.
   * @return parameters without additional information, i.e. only the keys listed in {@link #MAIN_PARAMETER_KEYS} and
   *         the parameters in {@link #KEY_ENTRIES} "withoutAdditionalInformation", will be set for each parameter.
   */
  public static AnySeq toAny(final List<ParameterDefinition> parameters, final boolean includingAdditionalAttributes) {
    final AnySeq paramSeq = DataFactory.DEFAULT.createAnySeq();
    for (final ParameterDefinition parameter : parameters) {
      paramSeq.add(parameter.toAny(includingAdditionalAttributes));
    }
    return paramSeq;
  }

  /** {@inheritDoc} */
  @Override
  public AnyMap toAny(final boolean includingAdditionalAttributes) {
    final AnyMap origParam = super.toAny(true);
    if (includingAdditionalAttributes) {
      return origParam;
    } else {
      final AnyMap newParam = origParam.getFactory().createAnyMap();
      for (final String key : MAIN_PARAMETER_KEYS) {
        if (origParam.containsKey(key)) {
          newParam.put(key, origParam.get(key));
        }
      }
      if (origParam.containsKey(KEY_ENTRIES)) {
        newParam.put(KEY_ENTRIES, ParameterDefinition.toAny(_entries, includingAdditionalAttributes));
      }
      return newParam;
    }
  }

  /** parse parameter defintion from a AnySeq. */
  public static List<ParameterDefinition> parseParameters(final AnySeq parameterSequence)
    throws InvalidConfigException {
    final List<ParameterDefinition> parameters = new ArrayList<ParameterDefinition>(parameterSequence.size());
    for (final Any parameterAny : parameterSequence) {
      if (parameterAny.isMap()) {
        parameters.add(new ParameterDefinition(parameterAny.asMap()));
      } else {
        throw new InvalidConfigException("An element of a parameter definition list is not a map, but a "
          + parameterAny.getValueType());
      }

    }
    return parameters;
  }

  /** @return the parameter's type. null for "any". */
  public Any.ValueType getType() {
    return _type;
  }

  /**
   * @return external name of type, either {@link ValueType#name()} (in lower case), or {@link #TYPENAME_ANY}, if type
   *         is null.
   */
  public String getTypeName() {
    if (_type == null) {
      return TYPENAME_ANY;
    }
    return _type.name().toLowerCase();
  }

  /** is this an optional parameter? */
  public boolean isOptional() {
    return _isOptional;
  }

  /** is it allowed for this parameter to have multiple values? */
  public boolean isMulti() {
    return _isMulti;
  }

  /** list of allowed values, null if not defined or for map/any types. */
  public List<Value> getValues() {
    return _values;
  }

  /*** list of allowed entries for map parameters, null if not defined or for other types. */
  public List<ParameterDefinition> getEntries() {
    return _entries;
  }
}
