/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.osgi.service.component.ComponentContext;

/**
 * Implements the handling of HTTP requests to request the data object types list. <br>
 * 
 * URL pattern: <code>smila/jobmanager/dataobjecttypes/</code> <br>
 * 
 * Methods allowed: <code>GET</code>
 */
public class DataObjectTypesHandler extends JsonRequestHandler {

  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void activate(final ComponentContext context) {
    super.activate(context);
    _defPersistence = _jobManager.getDefinitionPersistence();
  }

  /**
   * clean up JobManager on service shutdown.
   * 
   * @param context
   *          OSGi service context.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final Map<String, Object> resultMap = new HashMap<String, Object>();
    final List<Map<String, Object>> dotList = new ArrayList<Map<String, Object>>();
    for (final String dataObjectTypeName : _defPersistence.getDataObjectTypes()) {
      final LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>();
      map.put("name", dataObjectTypeName);
      map.put("url", URLCreator.create(getRequestHost(), requestUri, dataObjectTypeName));
      dotList.add(map);
    }
    resultMap.put(DataObjectTypeDefinition.KEY_DATA_OBJECT_TYPE_DEFINITIONS, dotList);
    return resultMap;
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }
}
