/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.Date;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.DefinitionBase;
import org.eclipse.smila.jobmanager.InvalidConfigException;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.WorkflowDefinition;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.jobmanager.persistence.PersistenceException;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;
import org.osgi.service.component.ComponentContext;

/**
 * Implements the handling of HTTP requests to request the workflow list or to define new workflows. <br>
 * 
 * URL pattern: <code>smila/jobmanager/workflows/</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class WorkflowsHandler extends JsonRequestHandler {
  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void activate(final ComponentContext context) {
    super.activate(context);
    _defPersistence = _jobManager.getDefinitionPersistence();
  }

  /**
   * clean up JobManager on service shutdown.
   * 
   * @param context
   *          OSGi service context.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
    if ("GET".equals(method)) {
      final AnySeq workflowList = DataFactory.DEFAULT.createAnySeq();
      boolean returnDetails = false;
      if (inputRecord != null && inputRecord.getMetadata().containsKey(JobManagerConstants.KEY_RETURN_DETAILS)) {
        returnDetails =
          inputRecord.getMetadata().getBooleanValue(JobManagerConstants.KEY_RETURN_DETAILS).booleanValue();
      }
      for (final String workflowName : _defPersistence.getWorkflows()) {
        final AnyMap map = DataFactory.DEFAULT.createAnyMap();
        map.put("name", workflowName);
        map.put("url", URLCreator.create(getRequestHost(), requestUri, workflowName).toExternalForm());
        if (returnDetails && _defPersistence.getWorkflow(workflowName).getTimestamp() != null) {
          map.put(JobManagerConstants.TIMESTAMP,
            DataFactory.DEFAULT.createDateTimeValue(_defPersistence.getWorkflow(workflowName).getTimestamp()));
        }
        workflowList.add(map);
      }
      resultMap.put(WorkflowDefinition.KEY_WORKFLOWS, workflowList);
    } else { // POST
      if (inputRecord == null) {
        throw new IllegalArgumentException("Missing input, cannot create workflows from nothing.");
      }
      final AnyMap definition = inputRecord.getMetadata();
      // create timestamp for this workflow and store it with the data
      final Any timestamp = inputRecord.getFactory().createDateTimeValue(new Date());
      definition.put(DefinitionBase.KEY_TIMESTAMP, timestamp);
      // remove readOnly flag if present.
      definition.remove(DefinitionBase.KEY_READ_ONLY);
      final WorkflowDefinition createdWorkflow = WorkflowDefinition.parseWorkflow(definition);
      _defPersistence.addWorkflow(createdWorkflow);
      resultMap.put("name", createdWorkflow.getName());
      resultMap.put(JobManagerConstants.TIMESTAMP, timestamp);
      resultMap.put("url", URLCreator.create(getRequestHost(), requestUri, createdWorkflow.getName())
        .toExternalForm());
    }
    return resultMap;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException or InvalidConfigException (bad input format)
   * as well as result code 500 ("INTERNAL_SERVER_ERROR") for PersistenceException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof NotFoundHTTPResult) {
      // NotFoundHTTPResult before PersistenceException! (Because of ConfigNotFoundException.)
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof PersistenceException) {
      return HttpStatus.INTERNAL_SERVER_ERROR;
    } else if (ex instanceof InvalidConfigException || ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if ("POST".equals(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }
}
