/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/

package org.eclipse.smila.bulkbuilder.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.bulkbuilder.BulkbuilderConstants;
import org.eclipse.smila.jobmanager.util.DataSizeParser;
import org.eclipse.smila.utils.config.ConfigUtils;

/**
 * Helper class for reading bulk builder configuration.
 */
public class BulkbuilderConfig {

  /** Constant for the multiplier to convert from seconds to milliseconds. */
  public static final long MILLISECONDS_MULTIPLIER = 1000;

  /** name of property file. */
  private static final String FILENAME = "bulkbuilder.properties";

  /** property name for bulk size limit for incremental records. */
  private static final String PROP_LIMIT_SIZE = BulkbuilderConstants.TASK_PARAM_LIMIT_SIZE;

  /** property name for time limit while collecting data for a new bulk of incremental records. */
  private static final String PROP_LIMIT_TIME = BulkbuilderConstants.TASK_PARAM_LIMIT_TIME;

  /** property name for maximum no of parallel micro bulks that can be processed in parallel. */
  private static final String PROP_MAX_PARALLEL_MICRO_BULKS = "maxParallelMicroBulks";

  /** default value for size limit in incremental mode. */
  private static final String DEFAULT_LIMIT_SIZE = "10m";

  /** default value for time limit in incremental mode. */
  private static final long DEFAULT_LIMIT_TIME = 120;

  /** default value for maximum no of parallel micro bulks that can be processed in parallel. */
  private static final long DEFAULT_MAX_PARALLEL_MICRO_BULKS = -1; // unlimited

  /** local Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** size limit value of record bulk (incremental mode) in bytes. */
  private long _limitIncSize;

  /** time limit value of record bulk (incremental mode) in seconds. */
  private long _limitIncTime;

  /** maximum no of parallel micro bulks that can be processed in parallel. */
  private long _maxParallelMicroBulks;

  /** configuration properties. */
  private Properties _properties = new Properties();

  /**
   * create instance with default values.
   */
  public BulkbuilderConfig() {
    setFrom(new Properties()); // init with default values
  }

  /**
   * @return bulk time limit in seconds.
   */
  public long getLimitTime() {
    return _limitIncTime;
  }

  /**
   * @return bulk time limit in milliseconds.
   */
  public long getLimitTimeMillis() {
    return getLimitTime() * MILLISECONDS_MULTIPLIER;
  }

  /**
   * @return bulk size limit in bytes.
   */
  public long getLimitSize() {
    return _limitIncSize;
  }

  /**
   * @return maximum no of parallel micro bulks that can be processed in parallel
   */
  public long getMaxParallelMicroBulks() {
    return _maxParallelMicroBulks;
  }

  /**
   * read configuration from config file.
   * 
   * @throws IOException
   *           error reading the file or missing required properties.
   */
  public void readConfiguration() throws IOException {
    InputStream propStream = null;
    try {
      propStream = ConfigUtils.getConfigStream(BulkbuilderConstants.BUNDLE_ID, FILENAME);
      if (propStream != null) {
        final Properties properties = new Properties();
        properties.load(propStream);
        setFrom(properties);
      }
    } catch (final Exception ex) {
      throw new IOException("Could not find " + FILENAME + " in configuration area.", ex);
    } finally {
      IOUtils.closeQuietly(propStream);
    }
  }

  /**
   * set values from given properties.
   * 
   * @param properties
   *          properties
   */
  public void setFrom(final Properties properties) {
    _properties = properties;
    _limitIncSize = getDataSizeProperty(PROP_LIMIT_SIZE, DEFAULT_LIMIT_SIZE);
    _limitIncTime = getIntProperty(PROP_LIMIT_TIME, DEFAULT_LIMIT_TIME);
    _maxParallelMicroBulks = getIntProperty(PROP_MAX_PARALLEL_MICRO_BULKS, DEFAULT_MAX_PARALLEL_MICRO_BULKS);
  }

  /**
   * parse a datasize value from a property.
   * 
   * @param propName
   *          property name.
   * @param defaultValue
   *          default value.
   * @return parsed value of default if none can be parsed.
   */
  protected long getDataSizeProperty(final String propName, final String defaultValue) {
    final String propertyString = _properties.getProperty(propName);
    if (propertyString != null) {
      try {
        return DataSizeParser.parse(propertyString);
      } catch (final Exception ex) {
        if (_log.isWarnEnabled()) {
          _log.warn("Value " + propertyString + " of " + propName
            + " is not an data size value. Falling back to default " + defaultValue);
        }
      }
    }
    return DataSizeParser.parse(defaultValue);
  }

  /**
   * parse a integer value from a property.
   * 
   * @param propName
   *          property name.
   * @param defaultValue
   *          default value.
   * @return parsed value of default if none can be parsed.
   */
  private long getIntProperty(final String propName, final long defaultValue) {
    final String propertyString = _properties.getProperty(propName);
    if (propertyString != null) {
      try {
        return Long.parseLong(propertyString);
      } catch (final Exception ex) {
        if (_log.isWarnEnabled()) {
          _log.warn("Value " + propertyString + " of " + propName
            + " is not an integer value. Falling back to default " + defaultValue);
        }
      }
    }
    return defaultValue;
  }
}
