/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * - Juergen Schumacher (Attensity Europe GmbH) - initial API and implementation 
 * - Andreas Weber (Attensity Europe GmbH) - removed processing services as BPEL pipeline extensions
 **********************************************************************************************************************/
package org.eclipse.smila.processing.bpel;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.xml.namespace.QName;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.ode.utils.DOMUtils;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.ode.ODEConfigProperties;
import org.eclipse.smila.ode.ODEServer;
import org.eclipse.smila.ode.ODEServerException;
import org.eclipse.smila.ode.WebServiceContextFactory;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.zookeeper.ZooKeeperService;
import org.osgi.service.component.ComponentContext;
import org.w3c.dom.Element;

/**
 * SMILA Workflow Processor that uses the Apache ODE BPEL engine to orchestrate SMILA pipelets in BPEL processes.
 * 
 * @author jschumacher
 * 
 */
public class ODEWorkflowProcessor implements WorkflowProcessor {
  /** the store to persist the workflow data. */
  public static final String WORKFLOW_STORE = "bpel";

  /** error message used to notify about (potential) inconsistent cluster state after a workflow (un)deploy failure. */
  private static final String ERROR_CLUSTER_INCONSISTENT =
    " There may be inconsistencies in cluster deploy of this workflow. Please repeat the operation.";

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** configuration properties. */
  private Properties _properties;

  /** BPEL server. */
  private ODEServer _bpelServer;

  /** Context factory for ODE server. */
  private WebServiceContextFactory _processingContext;

  /** to store the added/updated workflow definitions. */
  private WorkflowStorage _workflowStorage;

  /** Workflow deployment manager for the ODEServer. */
  private ODEDeploymentManager _deploymentManager;

  /** coordination of workflow updates in a SMILA cluster. */
  private WorkflowUpdateWatcher _updateWatcher;

  /** objectstore service reference. */
  private ObjectStoreService _objectStoreService;

  /** cluster config service reference. */
  private ClusterConfigService _clusterConfigService;

  /** ZooKeeper service reference. */
  private ZooKeeperService _zkService;

  /** Request ID sequence. */
  private final AtomicLong _requestIdSequence = new AtomicLong(0);

  /** the blackboards currently in use associated to the request IDs. */
  private final Map<String, Blackboard> _blackboards = new Hashtable<String, Blackboard>();

  /** last exception thrown in pipelet execution. To be reused in case the pipeline fails. */
  private final Map<String, Exception> _pipeletExceptions = new HashMap<String, Exception>();

  /** helper to create and parse XML DOM messages coming in and out of ODE engine. */
  private MessageHelper _messageHelper;

  /** performance counters for measurement of pipeline performance. */
  private final Map<String, PipelinePerformanceCounter> _pipelinePerformanceCounter =
    new HashMap<String, PipelinePerformanceCounter>();

  /*** process methods use read lock, deactivate needs write lock. */
  private final ReadWriteLock _lock = new ReentrantReadWriteLock(true);

  /** create processor. BPEL server is initialized in activate method. */
  public ODEWorkflowProcessor() {
    _log.debug(getClass().getName() + " instance created.");
  }

  /**
   * @see org.eclipse.smila.processing.WorkflowProcessor#process(java.lang.String, org.eclipse.smila.datamodel.id.Id[])
   */
  @Override
  public String[] process(final String workflowName, final Blackboard blackboard, final String[] recordIds)
    throws ProcessingException {
    final long startTime = System.nanoTime();
    final String requestId = initRequest(blackboard);
    String[] resultIds = null;
    boolean success = false;
    final PipelinePerformanceCounter counter = _pipelinePerformanceCounter.get(workflowName);
    try {
      if (_bpelServer == null) {
        throw new ProcessingException("Cannot process request, because BPEL engine is not yet initialised");
      }
      final Element message = _messageHelper.createMessage(blackboard, recordIds, requestId);
      if (_log.isTraceEnabled()) {
        _log.trace("Request: " + DOMUtils.domToString(message));
      }
      final QName processQName = getProcessId(workflowName);
      try {
        final Element result = _bpelServer.invoke(processQName, BPELConstants.OPERATION_NAME, message);
        if (result != null) {
          if (_log.isTraceEnabled()) {
            _log.trace("Final Result: " + DOMUtils.domToString(result));
          }
          resultIds = _messageHelper.parseMessage(blackboard, result);
          success = true;
          return resultIds;
        }
        return null;
      } catch (final ODEServerException ex) {
        final Exception pipeletEx = _pipeletExceptions.get(requestId);
        final String details = pipeletEx == null ? ex.getMessage() : pipeletEx.getMessage();
        final Exception cause = pipeletEx == null ? ex : pipeletEx;
        throw new ProcessingException("Error processing BPEL workflow " + workflowName + ": " + details, cause);
      }
    } catch (final ProcessingException ex) {
      countError(ex, false, counter);
      throw ex;
    } catch (final Throwable ex) {
      countError(ex, true, counter);
      throw new ProcessingException(ex);
    } finally {
      cleanupRequest(requestId);
      countInvocation(recordIds, resultIds, startTime, success, counter);
    }
  }

  /**
   * get the pipeline names of the active BPEL processes. The pipeline name is the local part of the EPR service name.
   * 
   * @return pipeline names of the active BPEL processes, or null, if engine is not active yet.
   */
  @Override
  public List<String> getWorkflowNames() {
    if (_pipelinePerformanceCounter == null) {
      return null;
    }
    return new ArrayList<String>(_pipelinePerformanceCounter.keySet());

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public AnyMap getWorkflowDefinition(final String workflowName) throws ProcessingException {
    final QName processId = getProcessId(workflowName);
    try {
      if (_deploymentManager.isPredefinedWorkflow(workflowName)) {
        final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
        final String bpelContent = _bpelServer.getBpelDocument(processId);
        if (bpelContent != null) {
          final Value bpel = DataFactory.DEFAULT.createStringValue(bpelContent);
          resultMap.put(WORKFLOW_NAME, workflowName);
          resultMap.put(WORKFLOW_READONLY, true);
          resultMap.put(WORKFLOW_DEFINITION, bpel);
          return resultMap;
        }
      } else if (_deploymentManager.isCustomWorkflow(workflowName)) {
        return _workflowStorage.getWorkflow(workflowName);
      }
    } catch (final Exception ex) {
      throw new ProcessingException("Error reading BPEL definition for workflow '" + workflowName + "'", ex);
    }
    return null;
  }

  @Override
  public void setWorkflowDefinition(final String workflowName, final AnyMap workflowDefinition)
    throws ProcessingException {
    checkWorkflowName(workflowName);
    workflowDefinition.remove(WorkflowProcessor.WORKFLOW_READONLY);
    final String timestamp = setTimestamp(workflowDefinition);
    if (StringUtils.isEmpty(workflowDefinition.getStringValue(WORKFLOW_DEFINITION))) {
      throw new ProcessingException("Workflow '" + workflowName + "' can not be deployed, " + WORKFLOW_DEFINITION
        + " is empty");
    }
    // validate first to avoid errors in following deploy.
    final File deployDir = _deploymentManager.validateWorkflow(workflowName, workflowDefinition);

    final AnyMap oldWorkflowAny = _workflowStorage.getWorkflow(workflowName); // needed for possible rollback

    // if objectstore update fails: no problem, local+cluster continue with old version.
    _workflowStorage.setWorkflow(workflowName, workflowDefinition);

    try {
      _updateWatcher.workflowUpdated(workflowName, timestamp);
    } catch (final ProcessingException e) {
      String errorMessage = "Error updating workflow '" + workflowName + "'.";
      _log.warn(errorMessage + " Error while notifying update watcher.", e);
      // ZK update failed: local+cluster still working with old workflow version,
      // but new version is stored in objectstore, try to rollback:
      try {
        if (oldWorkflowAny != null) {
          _workflowStorage.setWorkflow(workflowName, oldWorkflowAny);
        } else {
          _workflowStorage.deleteWorkflow(workflowName);
        }
      } catch (final ProcessingException e2) {
        errorMessage += ERROR_CLUSTER_INCONSISTENT;
        _log.error(errorMessage + " Error rolling back storage of new workflow version.", e2);
      }
      throw new ProcessingException(errorMessage, e, true); // recoverable
    }

    try {
      final QName processName = _deploymentManager.deployWorkflowDir(workflowName, deployDir);
      registerPipeline(processName);
    } catch (final ProcessingException e) {
      // ODE deploy failed: Houston, we got a problem. But due to former validation this should not happen.
      final String errorMessage = "Error updating workflow '" + workflowName + "'." + ERROR_CLUSTER_INCONSISTENT;
      _log.error(errorMessage + " Error deploying new workflow version.", e);
      throw new ProcessingException(errorMessage, e, true); // recoverable (?)
    }
  }

  @Override
  public void deleteWorkflowDefinition(final String workflowName) throws ProcessingException {
    checkWorkflowName(workflowName);
    final AnyMap oldWorkflowAny = _workflowStorage.getWorkflow(workflowName); // needed for possible rollback

    // if objectstore delete fails: no problem, local+cluster continue with old version.
    _workflowStorage.deleteWorkflow(workflowName);

    try {
      _updateWatcher.workflowDeleted(workflowName);
    } catch (final ProcessingException e) {
      String errorMessage = "Error deleting workflow '" + workflowName + "'.";
      _log.warn(errorMessage + " Error while notifying update watcher.", e);
      // ZK update failed: local+cluster still working with old workflow version,
      // but workflow is deleted in objectstore, try to rollback:
      try {
        if (oldWorkflowAny != null) {
          _workflowStorage.setWorkflow(workflowName, oldWorkflowAny);
        }
      } catch (final ProcessingException e2) {
        errorMessage += ERROR_CLUSTER_INCONSISTENT;
        _log.error(errorMessage + " Error rolling back deletion of workflow.", e2);
      }
      throw new ProcessingException(errorMessage, e, true); // recoverable
    }

    try {
      final QName processName = _deploymentManager.undeployWorkflow(workflowName);
      unregisterPipeline(processName);
    } catch (final ProcessingException e) {
      // ODE undeploy failed: local+cluster (potentially) would have different states, but that should not happen
      final String errorMessage = "Error deleting workflow '" + workflowName + "'." + ERROR_CLUSTER_INCONSISTENT;
      _log.error(errorMessage + " Error undeploying workflow.", e);
      throw new ProcessingException(errorMessage, e, true); // recoverable (?)
    }
  }

  @Override
  public void synchronizeWorkflowDefinition(final String workflowName, final boolean isDeleted)
    throws ProcessingException {
    checkWorkflowName(workflowName);
    final AnyMap workflowDefinition = _workflowStorage.getWorkflow(workflowName);
    if (isDeleted) {
      final QName processName = _deploymentManager.undeployWorkflow(workflowName);
      unregisterPipeline(processName);
    } else {
      if (workflowDefinition == null) {
        throw new ProcessingException("Definition for workflow '" + workflowName + "' not found for reload.");
      }
      final QName processName = _deploymentManager.deployWorkflow(workflowName, workflowDefinition);
      registerPipeline(processName);
    }
  }

  /**
   * get blackboard service for request.
   * 
   * @param id
   *          request ID
   * @return blackboard service.
   * @throws ProcessingException
   *           no blackboard associated with id
   */
  public Blackboard getBlackboard(final String id) throws ProcessingException {
    final Blackboard blackboard = _blackboards.get(id);
    if (blackboard == null) {
      throw new ProcessingException("Blackboard for request " + id + " is not registered anymore.");
    }
    return blackboard;
  }

  /**
   * store a pipelet exception for better error reporting if the engine finally fails.
   * 
   * @param requestId
   *          id of request
   * @param ex
   *          exception thrown during pipelet execution.
   */
  public void setPipeletException(final String requestId, final Exception ex) {
    _pipeletExceptions.put(requestId, ex);
  }

  /**
   * @return message helper that converts Processor/SearchMessages to XML and vice versa
   */
  public MessageHelper getMessageHelper() {
    return _messageHelper;
  }

  /**
   * OSGi Declarative Services service activation method. Initializes BPEL engine.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void activate(final ComponentContext context) {
    _lock.writeLock().lock();
    try {
      _workflowStorage = new WorkflowStorage(WORKFLOW_STORE, _objectStoreService, _clusterConfigService);
      if (_bpelServer == null) {
        readConfiguration();
        _messageHelper = new MessageHelper(_properties);
        _updateWatcher = new WorkflowUpdateWatcher(_zkService, this);
        initializeBPEL();
        _updateWatcher.startPolling();
        _updateWatcher.startWatching();
      }
    } catch (final Throwable ex) {
      // necessary to prevent automatic restarts of service before problem can be fixed.
      _log.error("Start of BPEL workflow service failed: Unknown fatal error. "
        + "Service is non-functional, please fix problem and restart bundle", ex);
    } finally {
      _lock.writeLock().unlock();
    }
  }

  /**
   * OSGi Declarative Services service deactivation method. Shuts down BPEL engine.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void deactivate(final ComponentContext context) {
    _lock.writeLock().lock();
    try {
      if (_updateWatcher != null) {
        _updateWatcher.stopPolling();
        _updateWatcher.stopWatching();
      }
      if (_bpelServer != null) {
        _bpelServer.shutdown();
        _bpelServer = null;
        _processingContext = null;
      }
    } finally {
      _lock.writeLock().unlock();
    }
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStore) {
    _objectStoreService = objectStore;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStore
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStore) {
    if (_objectStoreService == objectStore) {
      _objectStoreService = null;
    }
  }

  /**
   * set new {@link ClusterConfigService}. To be called by DS runtime before activation.
   * 
   * @param ccs
   *          new {@link ClusterConfigService}
   */
  public void setClusterConfigService(final ClusterConfigService ccs) {
    _clusterConfigService = ccs;
  }

  /**
   * remove an {@link ClusterConfigService}. To be called by DS runtime after deactivation.
   * 
   * @param ccs
   *          new {@link ClusterConfigService}
   */
  public void unsetClusterConfigService(final ClusterConfigService ccs) {
    if (_clusterConfigService == ccs) {
      _clusterConfigService = null;
    }
  }

  /**
   * set new {@link ZooKeeperService}. To be called by DS runtime before activation.
   * 
   * @param zks
   *          new {@link ZooKeeperService}
   */
  public void setZooKeeperService(final ZooKeeperService zks) {
    _zkService = zks;
  }

  /**
   * remove an {@link ZooKeeperService}. To be called by DS runtime after deactivation.
   * 
   * @param zks
   *          new {@link ZooKeeperService}
   */
  public void unsetZooKeeperService(final ZooKeeperService zks) {
    if (_zkService == zks) {
      _zkService = null;
    }
  }

  /**
   * initialize BPEL engine, initial deploy of existing BPEL pipelines. *
   * 
   * @throws IOException
   *           error reading the configuration
   * @throws ODEServerException
   *           error initializing the ODE server
   */
  private void initializeBPEL() throws IOException, ODEServerException {
    _log.debug("Initialize BPEL engine");
    final ODEConfigProperties odeConfig =
      new ODEConfigProperties(_properties, ConfigurationConstants.PROP_PREFIX_ODE);
    _processingContext = new WebServiceContextFactory();
    _bpelServer = new ODEServer(odeConfig, _processingContext);
    _bpelServer.registerExtensionBundle(new SMILAExtensionBundle());
    _deploymentManager = new ODEDeploymentManager(_bpelServer);
    try {
      deployPredefinedPipelines();
    } catch (final ProcessingException ex) {
      _log.error("Deployment of predefined pipelines failed. Installing custom pipelines should still work.", ex);
    }
    try {
      deployCustomPipelines();
    } catch (final ProcessingException ex) {
      _log.error("Deployment of existing custom pipelines failed. "
        + "Installing additional custom pipelines should still work.", ex);
    }
    _log.debug("Initialization of BPEL engine successful");
  }

  /** @return process id for workflowname. */
  private QName getProcessId(final String workflowName) {
    return new QName(NAMESPACE_PROCESSOR, workflowName);
  }

  /**
   * deploy predefined BPEL pipelines.
   * 
   * @throws ProcessingException
   *           error initializing pipelines.
   * @throws IOException
   *           error creating deployment directory.
   */
  private void deployPredefinedPipelines() throws ProcessingException, IOException {
    final String pipelineDirName =
      _properties
        .getProperty(ConfigurationConstants.PROP_PIPELINE_DIR, ConfigurationConstants.DEFAULT_PIPELINE_DIR);
    final Collection<QName> processes = _deploymentManager.deployPredefinedWorkflows(pipelineDirName);
    // JAXB used for parsing pipelet configs from BPEL needs this to be set.
    final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(getClass().getClassLoader());
    for (final QName processName : processes) {
      _log.info("Registering predefined pipeline " + processName);
      registerPipeline(processName);
    }
    Thread.currentThread().setContextClassLoader(oldCL);
  }

  /**
   * deploy BPEL pipelines stored in objectstore.
   * 
   * @throws ProcessingException
   *           error initializing pipelines.
   * @throws IOException
   *           error creating deployment directory.
   */
  private void deployCustomPipelines() throws ProcessingException, IOException {
    final Collection<String> workflowNames = _workflowStorage.getWorkflowNames();
    // JAXB used for parsing pipelet configs from BPEL needs this to be set.
    final ClassLoader oldCL = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(getClass().getClassLoader());
    for (final String workflowName : workflowNames) {
      final AnyMap workflowDefinition = _workflowStorage.getWorkflow(workflowName);
      final String timestamp = workflowDefinition.getStringValue(WORKFLOW_TIMESTAMP);
      final QName processName = _deploymentManager.deployWorkflow(workflowName, workflowDefinition);
      _updateWatcher.workflowLoadedOnStart(workflowName, timestamp);
      _log.info("Registering custom pipeline " + processName);
      registerPipeline(processName);
    }
    Thread.currentThread().setContextClassLoader(oldCL);
  }

  /** register pipeline process name. */
  private void registerPipeline(final QName processName) {
    PipeletManager.getInstance().registerPipeline(this, processName);
    final String pipelineName = processName.getLocalPart();
    _pipelinePerformanceCounter.put(pipelineName, new PipelinePerformanceCounter(pipelineName));
  }

  /** unregister pipeline process name. */
  private void unregisterPipeline(final QName processName) {
    if (processName != null) {
      PipeletManager.getInstance().registerPipeline(this, processName);
      final String pipelineName = processName.getLocalPart();
      _pipelinePerformanceCounter.remove(pipelineName);
    }
  }

  /**
   * @throws ProcessingException
   *           if workflowName is null or refers to a predefined workflow.
   */
  private void checkWorkflowName(final String workflowName) throws ProcessingException {
    if (StringUtils.isEmpty(workflowName)) {
      throw new ProcessingException("Workflow name must not be undefined or empty.");
    }
    if (_deploymentManager.isPredefinedWorkflow(workflowName)) {
      throw new ProcessingException("Workflow '" + workflowName + "' can not be changed, cause it's predefined");
    }
  }

  /** check if workflow definition has a timestamp and create one, if not. */
  private String setTimestamp(final AnyMap workflowDefinition) {
    final Value value = workflowDefinition.getFactory().createDateTimeValue(new Date());
    workflowDefinition.put(WorkflowProcessor.WORKFLOW_TIMESTAMP, value);
    return value.asString();
  }

  /**
   * read configuration property file.
   * 
   * @throws IOException
   *           error reading configuration file
   */
  private void readConfiguration() throws IOException {
    _properties = new Properties();
    InputStream configurationFileStream = null;
    try {
      configurationFileStream =
        ConfigUtils.getConfigStream(ConfigurationConstants.BUNDLE_NAME, ConfigurationConstants.CONFIGURATION_FILE);
      _properties.load(configurationFileStream);
    } catch (final IOException ex) {
      throw new IOException("Could not read configuration property file "
        + ConfigurationConstants.CONFIGURATION_FILE + ": " + ex.toString());
    } finally {
      IOUtils.closeQuietly(configurationFileStream);
    }
  }

  /**
   * generate new request id and store blackboard.
   * 
   * @param blackboard
   *          request blackboard.
   * @return new request Id
   */
  private String initRequest(final Blackboard blackboard) {
    _lock.readLock().lock();
    final String requestId = Long.toString(_requestIdSequence.getAndIncrement());
    if (_log.isDebugEnabled()) {
      _log.debug("Starting to process request ID = " + requestId);
    }
    _blackboards.put(requestId, blackboard);
    return requestId;
  }

  /** add statistics about a completed invocation to the pipeline counter. */
  private void countInvocation(final String[] incomingIds, final String[] outgoingIds, final long startTime,
    final boolean success, final PipelinePerformanceCounter counter) {
    if (counter != null) {
      counter.countInvocationNanos(System.nanoTime() - startTime, success, ArrayUtils.getLength(incomingIds),
        ArrayUtils.getLength(outgoingIds));
    }
  }

  /** add an error to the pipeline counter. */
  private void countError(final Throwable ex, final boolean isCritical, final PipelinePerformanceCounter counter) {
    if (counter != null) {
      counter.addError(ex, isCritical);
    }
  }

  /**
   * release blackboard and caught exception.
   * 
   * @param requestId
   *          request Id
   */
  private void cleanupRequest(final String requestId) {
    if (_log.isDebugEnabled()) {
      _log.debug("Cleaning up request ID = " + requestId);
    }
    _blackboards.remove(requestId);
    _pipeletExceptions.remove(requestId);
    _lock.readLock().unlock();
  }

}
