package org.eclipse.smila.processing.bpel;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.TreeSet;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.clusterconfig.ClusterConfigException;
import org.eclipse.smila.clusterconfig.ClusterConfigService;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.ipc.IpcSerializationUtils;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.objectstore.StoreObject;
import org.eclipse.smila.objectstore.StoreOutputStream;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;
import org.eclipse.smila.processing.ProcessingException;

/**
 * Helper class to store workflow definitions in objectstore.
 * 
 * @author aweber
 */
public final class WorkflowStorage {

  /** max retries for store requests on IOExceptions. */
  private static final int MAX_RETRY_ON_STORE_UNAVAILABLE = 3;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** for record<->BON/JSON serialization. */
  private final IpcSerializationUtils _serializationUtils = new IpcSerializationUtils();

  /** has store already been successfully prepared? */
  private boolean _isStorePrepared;

  /** store name. */
  private final String _storeName;

  /** objectstore reference for storing the workflows. */
  private final ObjectStoreService _objectStoreService;

  /** to get the failsafety configutation needed to create the store properties. */
  private final ClusterConfigService _clusterConfigService;

  /** */
  public WorkflowStorage(final String storeName, final ObjectStoreService objectStore,
    final ClusterConfigService clusterConfig) {
    _storeName = storeName;
    _objectStoreService = objectStore;
    _clusterConfigService = clusterConfig;
  }

  /** add/update workflow with given name. */
  public void setWorkflow(final String workflowName, final AnyMap workflow) throws ProcessingException {
    try {
      writeObjectToStore(workflowName, workflow);
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error adding workflow '" + workflowName + "' to store", e);
    }
  }

  /** delete workflow with given name. */
  public void deleteWorkflow(final String workflowName) throws ProcessingException {
    try {
      removeObjectFromStore(workflowName);
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error deleting workflow '" + workflowName + "' to store", e);
    }
  }

  /** return workflow with given name. */
  public AnyMap getWorkflow(final String workflowName) throws ProcessingException {
    try {
      if (!existsObjectInStore(workflowName)) {
        return null;
      }
      return readObjectFromStore(workflowName);
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error getting workflow '" + workflowName + "' from store", e);
    }
  }

  /** return names of all stored workflows. */
  public Collection<String> getWorkflowNames() throws ProcessingException {
    try {
      return readObjectNamesFromStore();
    } catch (final ObjectStoreException e) {
      throw new ProcessingException("Error getting workflow names from store", e);
    }
  }

  /**
   * @param objectId
   *          the object id under which the object is stored
   * @param object
   *          the object to store
   * @throws ObjectStoreException
   *           error writing object to the {@link ObjectStoreService}.
   */
  private void writeObjectToStore(final String objectId, final AnyMap object) throws ObjectStoreException {
    Exception retryableEx = null;
    ensureStore();
    for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
      StoreOutputStream storeOutputStream = null;
      try {
        storeOutputStream = _objectStoreService.writeObject(_storeName, objectId);
        _serializationUtils.map2JsonStream(storeOutputStream, object);
        return;
      } catch (final IOException ex) {
        if (storeOutputStream != null) {
          storeOutputStream.abort();
        }
        retryableEx = ex;
        _log.warn("IOException on writing object '" + objectId + "', retrying: " + ex.toString());
      } catch (final ServiceUnavailableException ex) {
        retryableEx = ex;
        _log.warn("ServiceUnavailableException on writing object '" + objectId + "', retrying: " + ex.toString());
      } finally {
        IOUtils.closeQuietly(storeOutputStream);
      }
    }
    throw new ServiceUnavailableException("Finally failed to write object '" + objectId + "' to store", retryableEx);
  }

  /**
   * @return the object names found in the store
   * @throws ObjectStoreException
   *           could not read object information from the {@link ObjectStoreService}.
   */
  private Collection<String> readObjectNamesFromStore() throws ObjectStoreException {
    ensureStore();
    final Collection<String> objectNames = new TreeSet<String>();
    for (final StoreObject info : _objectStoreService.getStoreObjectInfos(_storeName)) {
      final String objectName = info.getId();
      objectNames.add(objectName);
    }
    return objectNames;
  }

  /**
   * @param objectId
   *          the object id of the object to read from store
   * @return the stored object for the given object id
   * @throws ObjectStoreException
   *           could not read object from the {@link ObjectStoreService}.
   */
  private AnyMap readObjectFromStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    Exception retryableEx = null;
    for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
      try {
        InputStream jobJsonStream = null;
        try {
          jobJsonStream = _objectStoreService.readObject(_storeName, objectId);
          return _serializationUtils.jsonStream2record(jobJsonStream).getMetadata();
        } finally {
          IOUtils.closeQuietly(jobJsonStream);
        }
      } catch (final IOException ex) {
        retryableEx = ex;
        _log.warn("IOException on writing object '" + objectId + "', retrying: " + ex.toString());
      } catch (final ServiceUnavailableException ex) {
        retryableEx = ex;
        _log.warn("ServiceUnavailableException on writing object '" + objectId + "', retrying: " + ex.toString());
      }
    }
    throw new ServiceUnavailableException("Finally failed to read object '" + objectId + "' from store",
      retryableEx);
  }

  /**
   * Removes an object from the store.
   * 
   * @param objectId
   *          the id of the object
   * @throws ObjectStoreException
   *           object could not be removed
   */
  private boolean existsObjectInStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    return ObjectStoreRetryUtil.retryExistsObject(_objectStoreService, _storeName, objectId);
  }

  /**
   * Removes an object from the store.
   * 
   * @param objectId
   *          the id of the object
   * @throws ObjectStoreException
   *           object could not be removed
   */
  private void removeObjectFromStore(final String objectId) throws ObjectStoreException {
    ensureStore();
    _objectStoreService.removeObject(_storeName, objectId);
  }

  /**
   * Makes sure the store where the workflow data is stored, does exist and creates it with the store properties
   * retrieved from {@link ClusterConfigService} if it does not exist, yet.
   * 
   * @throws ObjectStoreException
   *           the store could not be created.
   * 
   */
  private void ensureStore() throws ObjectStoreException {
    if (!_isStorePrepared) {
      final AnyMap storeProps = getStoreProperties();
      ServiceUnavailableException ioex = null;
      for (int i = 0; i < MAX_RETRY_ON_STORE_UNAVAILABLE; i++) {
        try {
          synchronized (this) {
            if (!_objectStoreService.existsStore(_storeName)) {
              _objectStoreService.createStore(_storeName, storeProps);
            }
            _isStorePrepared = true;
            return;
          }
        } catch (final ServiceUnavailableException ex) {
          ioex = ex;
          _log.warn("ServiceUnavailableException on preparing store '" + _storeName + "', retrying: "
            + ex.toString());
        }
      }
      throw new ServiceUnavailableException("Finally failed to prepare store '" + _storeName + "'", ioex);
    }
  }

  /**
   * Determines the store properties from the failsafety level.
   * 
   * @return the store properties.
   */
  private AnyMap getStoreProperties() {
    // set the default replication level of RunStorage:
    int numberOfNodes = 1;
    long failSafetyLevel = 0;
    try {
      numberOfNodes = _clusterConfigService.getClusterNodes().size();
      failSafetyLevel = _clusterConfigService.getFailSafetyLevel();
    } catch (final ClusterConfigException e) {
      _log.warn("Can't connect to local ClusterConfigService");
    }
    // special handling:
    // - 1 node: CL=RL=0;
    // - 2 nodes: CL=0; RL=1;
    // - >2 nodes: CL=1; RL=CL+FSL;
    final long commitlevel;
    final long replicationlevel;
    if (numberOfNodes <= 2) {
      commitlevel = 0;
      replicationlevel = numberOfNodes - 1;
    } else {
      commitlevel = 1;
      replicationlevel = commitlevel + failSafetyLevel;
    }
    final AnyMap storeProperties = DataFactory.DEFAULT.createAnyMap();
    storeProperties.put("replicationlevel", replicationlevel);
    storeProperties.put("commitlevel", commitlevel);
    return storeProperties;
  }

}
