/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.embedded.EmbeddedSolrServer;
import org.apache.solr.client.solrj.impl.BinaryRequestWriter;
import org.apache.solr.client.solrj.impl.CommonsHttpSolrServer;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.apache.solr.core.CoreContainer;
import org.eclipse.smila.solr.admin.SolrAdmin;
import org.eclipse.smila.solr.admin.SolrEmbeddedAdmin;
import org.eclipse.smila.solr.admin.SolrHttpAdmin;
import org.eclipse.smila.solr.util.SolrHelper;
import org.eclipse.smila.solr.util.SolrUtils;
import org.xml.sax.SAXException;

/**
 * The SolrManager class.
 * 
 * @author pwissel
 * 
 */
public class SolrServerManager {

  /**
   * The SolrProperties.
   */
  private final SolrProperties _properties = new SolrProperties();

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(SolrServerManager.class);

  /**
   * Synchronized map which hold all running servers.
   */
  private final Map<String, SolrServer> _servers = Collections.synchronizedMap(new HashMap<String, SolrServer>());

  /**
   * The core container (only for embedded use).
   */
  private CoreContainer _container;

  /**
   * The SolrAdmin.
   */
  private SolrAdmin _admin;

  /**
   * The SolrHelper.
   */
  private SolrHelper _helper;

  /**
   * 
   */
  public SolrServerManager() {
    this(new SolrProperties());
  }

  /**
   * Default constructor.
   */
  public SolrServerManager(SolrProperties properties) {
    try {
      if (_log.isDebugEnabled()) {
        _log.debug("Start SolrManager instance...");
      }

      _helper = new SolrHelper(properties);

      if (_properties.isEmbedded()) {
        loadCoreContainer();
        _admin = new SolrEmbeddedAdmin(_helper, _container);
        return;
      }
      _admin = new SolrHttpAdmin(_helper);

    } catch (Exception exception) {
      if (_log.isErrorEnabled()) {
        _log.error("Error while initialize SolrManager", exception);
      }
    }
  }

  /**
   * Load CoreContainer configuration file form solr.home (solr.xml).
   * 
   * @throws Exception
   * 
   * @throws ParserConfigurationException
   *           ParserConfigurationException.
   * @throws IOException
   *           IOException.
   * @throws SAXException
   *           SAXException.
   */
  private void loadCoreContainer() throws Exception {
    try {
      final File solr = _helper.getSolrXML();
      _container = new CoreContainer();
      _container.load(_helper.getSolrHome().getAbsolutePath(), solr);

      if (_log.isInfoEnabled()) {
        _log.info("Solr CoreContainer for EmbeddedSolrServer loaded successfully.");
      }

      if (_log.isDebugEnabled()) {
        _log.debug("Available solr cores: " + StringUtils.join(_container.getCoreNames(), File.pathSeparator));
      }
    } catch (Exception exception) {
      throw new Exception("Error while load CoreContainer", exception);
    }
  }

  /**
   * Get the SolrProperties.
   * 
   * @return the SolrProperties.
   */
  public SolrProperties getSolrProperties() {
    return _properties;
  }

  /**
   * Get the SolrAdmin.
   * 
   * @return the SolrAdmin.
   */
  public SolrAdmin getSolrAdmin() {
    return _admin;
  }

  /**
   * Get the SolrHelper.
   * 
   * @return the SolrHelper.
   */
  public SolrHelper getSolrHelper() {
    return _helper;
  }

  /**
   * Get CoreContainer.
   * 
   * @return the core container.
   */
  public CoreContainer getCoreContainer() {
    return _container;
  }

  /**
   * Get a SolrServer by core name.
   * 
   * @param coreName
   *          the core name.
   * @return the SolrServer.
   * @throws SolrServerException
   *           SolrServerException.
   * @throws IOException
   *           IOException.
   */
  public SolrServer getSolrServer(String coreName) throws SolrServerException, IOException {
    if (_properties.isEmbedded()) {
      return getEmbeddedSolrServer(coreName);
    }
    return getHttpSolrServer(coreName);
  }

  /**
   * Get or start an EmbeddedSolrServer by core name.
   * 
   * @param coreName
   *          the core name.
   * @return the EmbeddedSolrServer.
   * @throws SolrServerException
   *           SolrServerException.
   */
  private SolrServer getEmbeddedSolrServer(String coreName) throws SolrServerException {
    SolrServer server = _servers.get(coreName);
    if (server == null) {
      if (!_container.getCoreNames().contains(coreName)) {
        throw new SolrServerException("No configuration found in solr.home '" + _container.getSolrHome()
          + "' for core with name: " + coreName);
      }
      server = new EmbeddedSolrServer(_container, coreName);
      _servers.put(coreName, server);
    }
    return server;
  }

  /**
   * Get a HttpSolrServer by core name. Take the server URL from properties or default one.
   * 
   * @param coreName
   *          the core name.
   * @return the HttpSolrServer.
   * @throws SolrServerException
   *           SolrServerException.
   * @throws IOException
   *           IOException.
   */
  private SolrServer getHttpSolrServer(String coreName) throws SolrServerException, IOException {
    SolrServer server = _servers.get(coreName);
    if (server == null) {
      final String url = _properties.getServerUrl();
      server = getHttpSolrServer(url, coreName);
      _servers.put(coreName, server);
    }
    return server;
  }

  /**
   * Get a HttpSolrServer by core name and server URL.
   * 
   * @param url
   *          the server URL.
   * @param coreName
   *          the core name.
   * @return the HttpSolrServer.
   * @throws SolrServerException
   *           SolrServerException.
   * @throws IOException
   *           IOException.
   */
  private SolrServer getHttpSolrServer(String url, String coreName) throws SolrServerException, IOException {
    if (coreName != null) {
      url = url + "/" + coreName;
    }
    final CommonsHttpSolrServer server = new CommonsHttpSolrServer(url);
    server.setRequestWriter(new BinaryRequestWriter());
    final SolrPingResponse ping = server.ping();
    if (SolrUtils.responseStatusIsError(ping)) {
      final String msg = MessageFormat.format("Cannot ping given server for URL: {0}. Response is: {1} {2}", //
        ping.getRequestUrl(), ping.getStatus(), ping.toString());
      throw new SolrServerException(msg);
    }
    return server;
  }

}
