/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import java.text.MessageFormat;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.FacetField.Count;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.SpellCheckResponse;
import org.apache.solr.client.solrj.response.SpellCheckResponse.Suggestion;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.util.NamedList;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.search.api.helper.ResultBuilder;
import org.eclipse.smila.solr.SolrConstants;

/**
 * SolrResultBuilder.
 * 
 * @author pwissel
 * 
 */
public class SolrResultBuilder extends ResultBuilder {

  /**
   * Default workflow name.
   */
  private static final String DEFAULT_WORKFLOW = "SolrSearchDefaultWorkflow";

  /**
   * The QueryResponse from solr server.
   */
  private final QueryResponse _response;

  /**
   * The log.
   */
  final Log _log = LogFactory.getLog(this.getClass());

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param record
   *          the query record.
   * @param response
   *          the solr query response.
   */
  public SolrResultBuilder(Record record, QueryResponse response) {
    super(DEFAULT_WORKFLOW, record);
    _response = response;
  }

  /**
   * Constructor.
   * 
   * @param workflowName
   *          the workflow name.
   * @param record
   *          the query record.
   * @param response
   *          the solr query response.
   */
  public SolrResultBuilder(String workflowName, Record record, QueryResponse response) {
    super(workflowName, record);
    _response = response;
  }

  /**
   * Process the response result and add to query record.
   * 
   * @return
   */
  public Record processResponse() {
    // set runtime
    setRuntime(new Long(_response.getQTime()));

    processQueryResponse();
    processFacetResponse();
    processTermsResponse();
    processSpellcheckResponse();

    return getResult();
  }

  /**
   * Process query response.
   */
  private void processQueryResponse() {
    final SolrDocumentList results = _response.getResults();
    if (results == null) {
      return;
    }

    // set count and max score
    setCount(results.getNumFound());
    setMaxScore(results.getMaxScore());

    // iterate over result documents
    for (SolrDocument document : results) {

      // get document (record) id
      final String id = (String) document.getFieldValue(SolrConstants.CORE_FIELD_ID);

      // get score
      Double score = -1.0;
      final Object scoreObj = document.getFieldValue(SolrConstants.CORE_FIELD_SCORE);
      try {
        final Number scoreNumber = (Number) scoreObj;
        score = scoreNumber.doubleValue();
      } catch (Exception e) {
        if (_log.isWarnEnabled()) {
          final String msg;
          if (e instanceof NullPointerException) {
            msg = "No score value in the solr result for id: " + id;
          } else if (e instanceof ClassCastException) {
            msg =
              MessageFormat.format(
                "The score value returned from solr is not a Number for id: {0} -> score as string: {1}", id,
                scoreObj);
          } else {
            msg = "Cannot get score for id: " + id;
          }
          _log.warn(msg + ". Setting score to -1. Check your config to fix this.");
        }
      }

      // create result item and add document fields
      final AnyMap item = addResultItem(id, score);
      addResultFieldsToItem(document, item);
      processHighlightingResponse(item);
    }
  }

  /**
   * Add a result field (fl) to the resilt item.
   * 
   * @param document
   *          the solr document.
   * @param item
   *          the result item.
   */
  private void addResultFieldsToItem(final SolrDocument document, AnyMap item) {
    for (final Entry<String, Object> entry : document.entrySet()) {
      try {
        final String key = entry.getKey();
        if (key.equals(SolrConstants.CORE_FIELD_ID) || key.equals(SolrConstants.CORE_FIELD_SCORE)) {
          continue;
        }

        final Object value = entry.getValue();
        if (value != null) {

          if (value instanceof Collection) {
            addMultiKeyValuePairToItem(key, value, item);
          } else {
            addSingleKeyValuePairToItem(key, value, item);
          }

        }

      } catch (Exception exception) {
        if (_log.isWarnEnabled()) {
          _log.warn("Error adding solr result to record item for field name: " + entry.getKey()
            + ". this field has been skipped.", exception);
        }
      }
    }
  }

  /**
   * Add a single key value pair to result item.
   * 
   * @param key
   *          the key.
   * @param value
   *          the value object.
   * @param item
   *          the result item.
   */
  private void addSingleKeyValuePairToItem(String key, Object value, AnyMap item) {
    final DataFactory factory = item.getFactory();
    final Value autoConvertValue = factory.autoConvertValue(value);
    item.put(key, autoConvertValue);
  }

  /**
   * Add a multi value key value pair to result item as a sequence.
   * 
   * @param key
   *          the key.
   * @param value
   *          the values as list.
   * @param item
   *          the result item.
   */
  private void addMultiKeyValuePairToItem(String key, Object value, AnyMap item) {
    final DataFactory factory = item.getFactory();
    final Collection<?> multiValues = (Collection<?>) value;
    final AnySeq seq = factory.createAnySeq();
    item.put(key, seq);

    for (final Object multiValue : multiValues) {
      final Value autoConvertValue = factory.autoConvertValue(multiValue);
      seq.add(autoConvertValue);
    }
  }

  /**
   * Process the highlighting response for a record. Note, this also works when HL is configered in the solrconfig.xml
   * by defining default props for the handler.
   * 
   * @param item
   *          the result item.
   */
  private void processHighlightingResponse(AnyMap item) {
    final Map<String, Map<String, List<String>>> highlighting = _response.getHighlighting();
    if (highlighting == null) {
      return;
    }

    final String id = item.getStringValue("_recordid");
    final Map<String, List<String>> map = highlighting.get(id);
    for (String key : map.keySet()) {
      final List<String> list = map.get(key);
      if (list.size() > 1) {
        addMultiKeyValuePairToItem(key, list, item);
      } else {
        addSingleKeyValuePairToItem(key, list.get(0), item);
      }
    }

    // TODO: If there exist a result field with the same name as the highlighting field name, this field is overwritten
    // by default. May be make this configurable (overwrite or store highlighting field under specified name).
  }

  /**
   * Process facet response.
   */
  private void processFacetResponse() {
    if (_response.getFacetFields() != null) {
      for (FacetField facet : _response.getFacetFields()) {
        addFacetFieldToRecord(facet);
      }
    }
    if (_response.getFacetDates() != null) {
      for (FacetField facet : _response.getFacetDates()) {
        addFacetFieldToRecord(facet);
      }
    }
    if (_response.getFacetQuery() != null) {
      final Map<String, Integer> facetQuery = _response.getFacetQuery();
      if (facetQuery.size() > 0) {
        final Set<String> keySet = facetQuery.keySet();
        final String firstKey = keySet.toArray(new String[keySet.size()])[0];
        final String name = StringUtils.substringBefore(firstKey, ":");
        final AnySeq group = addGroup(name);
        for (String key : keySet) {
          addGroupValue(group, key, facetQuery.get(key).longValue());
        }
      }
    }
  }

  /**
   * Add a facet field to query record.
   * 
   * @param facet
   *          the facet field.
   */
  private void addFacetFieldToRecord(FacetField facet) {
    final List<Count> values = facet.getValues();
    if (values == null || values.size() < 1) {
      return;
    }
    final AnySeq group = addGroup(facet.getName());
    final DataFactory factory = group.getFactory();
    for (Count count : values) {
      final AnyMap map = addGroupValue(group, count.getName(), count.getCount());
      final String filterQuery = count.getAsFilterQuery();
      final Value value = factory.autoConvertValue(filterQuery);
      map.put(SolrConstants.FILTER_QUERY, value);
    }
    group.add(facet.getValueCount());
  }

  /**
   * Process terms response.
   */
  private void processTermsResponse() {
    final NamedList<?> terms = (NamedList<?>) _response.getResponse().get(SolrConstants.TERMS);
    if (terms == null) {
      return;
    }

    final AnyMap map = getTermsMap(true);
    final DataFactory factory = map.getFactory();

    for (int i = 0; i < terms.size(); i++) {
      final NamedList<?> field = (NamedList<?>) terms.getVal(i);

      for (int z = 0; z < field.size(); z++) {
        final String name = field.getName(z);
        final Object value = field.getVal(z);
        final Value autoConvertValue = factory.autoConvertValue(value);
        map.put(name, autoConvertValue);
      }
    }
  }

  /**
   * Process spellcheck result.
   */
  private void processSpellcheckResponse() {
    final SpellCheckResponse spellcheck = _response.getSpellCheckResponse();
    if (spellcheck == null) {
      return;
    }

    final Map<String, Suggestion> suggestions = spellcheck.getSuggestionMap();
    if (suggestions != null && suggestions.size() > 0) {
      final AnyMap map = getSpellcheckMap(true);

      // add suggestions
      for (final Entry<String, Suggestion> entry : suggestions.entrySet()) {
        final Suggestion suggestion = entry.getValue();
        if (suggestion != null) {
          final AnyMap results = map.getMap(entry.getKey(), true);
          final List<String> alternatives = suggestion.getAlternatives();
          final List<Integer> altervativeFrequencies = suggestion.getAlternativeFrequencies();
          for (int i = 0; i < suggestion.getNumFound(); i++) {
            if (altervativeFrequencies != null) {
              results.put(alternatives.get(i), altervativeFrequencies.get(i));
            } else {
              results.put(alternatives.get(i), -1);
            }
          }
        }
      }

      // add collation, if exists.
      final String collation = spellcheck.getCollatedResult();
      if (collation != null) {
        map.put(SolrConstants.COLLATION, collation);
      }

    }

  }

  /**
   * Get the solr result map (_solr.result).
   * 
   * @return the solr result map.
   */
  AnyMap getSolrResultMap(Boolean create) {
    return getResult().getMetadata().getMap(SolrConstants.RESULT_MAP, create);
  }

  /**
   * Get the terms map.
   * 
   * @param create
   *          true to create maps, false otherwise.
   * @return the terms map.
   */
  private AnyMap getTermsMap(Boolean create) {
    return getSolrResultMap(create).getMap(SolrConstants.TERMS, create);
  }

  /**
   * Get the spellcheck map.
   * 
   * @param create
   *          true to create maps, false otherwise.
   * @return the spellcheckmap.
   */
  private AnyMap getSpellcheckMap(Boolean create) {
    return getSolrResultMap(create).getMap(SolrConstants.SPELLCHECK, create);
  }

  /**
   * Set max score.
   * 
   * @param max
   *          score.
   */
  private void setMaxScore(Number maxScore) {
    getSolrResultMap(true).put(SolrConstants.MAX_SCORE, maxScore);
  }
}
