/*******************************************************************************
 * Copyright (c) 2010 Empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Thomas Menzel (brox IT Solution GmbH) new implementation loosely based on Igor Novakovic (Empolis GmbH)
 * work
 *******************************************************************************/

package org.eclipse.smila.solr.search;

import static org.apache.commons.lang.StringUtils.isBlank;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.request.QueryRequest;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;
import org.eclipse.smila.processing.Pipelet;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.solr.Activator;
import org.eclipse.smila.solr.SolrServerManager;

/**
 * This pipelet integrates the solr search via the SolrJ client using the binary protocol by default.
 */
public class SolrSearchPipelet implements Pipelet {

  /**
   * The log.
   */
  private final Log _log = LogFactory.getLog(SolrSearchPipelet.class);

  /**
   * The SolrServerManager.
   */
  private SolrServerManager _solrManager;

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.Pipelet#configure(org.eclipse.smila.datamodel.AnyMap)
   */
  @Override
  public void configure(final AnyMap config) throws ProcessingException {
    // nothing
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.processing.Pipelet#process(org.eclipse.smila.blackboard.Blackboard, java.lang.String[])
   */
  @Override
  public String[] process(final Blackboard blackboard, final String[] queryIds) throws ProcessingException {
    if (queryIds.length != 1) {
      /* LIMIT: | tmenzel @ Mar 21, 2011 */
      throw new ProcessingException("number of queryIds expected to be 1 but is: " + queryIds.length);
    }
    /*
     * BETTER: it should be possible to config some default properties and be able to distinguish them by name. a search
     * could reference these then and have them set for the current search, while still being able to override them
     * locally.
     * 
     * this can be done by loading such configs on startup and use the DefaultSolrParams to wrap the SolrQuery
     * 
     * | TM @ Feb 16, 2011
     */

    try {
      final String queryId = queryIds[0];
      final Record record = blackboard.getRecord(queryId);
      final SolrQueryParameterAccessor accessor = new SolrQueryParameterAccessor(blackboard, queryId);

      final String coreName = accessor.getIndexName();
      if (isBlank(coreName)) {
        _log.info("Core/index name missing from query or is blank. Skipping sesarch.");
        return queryIds;
      }
      final SolrServer solrServer = getSolrManager().getSolrServer(coreName);
      final SolrQuery solrQuery = new SolrQueryConverter(accessor).toSolrQuery();

      if (_log.isDebugEnabled()) {
        _log.debug("SolrQuery: " + solrQuery.toString());
      }

      final QueryResponse solrResponse;
      try {
        /*
         * PERF: since POST is always less performant than GET, we should calc/aprocimate the lenght of query and have a
         * config'able threshhold where we switch from GET to POST| TM @ Jun 23, 2011
         */
        final QueryRequest queryRequest = new QueryRequest(solrQuery, SolrRequest.METHOD.POST);
        solrResponse = queryRequest.process(solrServer);
      } catch (final Exception e) {
        final String msg = "Error while sending solr query: " + solrQuery.toString();
        _log.error(msg, e);
        throw new ProcessingException(msg, e);
      }

      final SolrResultBuilder resultBuilder = new SolrResultBuilder("SolrSearch", record, solrResponse);
      final Record resultRecord = resultBuilder.processResponse();

      if (_log.isTraceEnabled()) {
        _log.trace(XmlSerializationUtils.serialize2string(resultRecord));
      }

    } catch (final Exception e) {
      _log.error("error on processing query: " + queryIds[0], e);
      /*
       * BETTER: would be good to add the exception to the query record, so that it can be returned to the client. | TM
       * 
       * @ Feb 15, 2011
       */
      throw new ProcessingException(e);
    } // try
    return queryIds;
  }

  /**
   * Set SolrManager.
   * 
   * @param solrManager
   *          the SolrManager.
   */
  public void setSolrManager(SolrServerManager solrManager) {
    _solrManager = solrManager;
  }

  /**
   * Get SolrManager.
   * 
   * @return the SolrManager.
   */
  public SolrServerManager getSolrManager() {
    if (_solrManager == null) {
      _solrManager = Activator.getInstance().getSolrManager();
    }

    return _solrManager;
  }

}
