/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Dmitry Hazin (brox IT Solutions GmbH) - initial creator Sebastian Voigt (Brox IT Solutions GmbH) -
 * initial creator Juergen Schumacher (Attensity Europe GmbH) - update to new data model.
 **********************************************************************************************************************/
package org.eclipse.smila.blackboard;

import java.io.File;
import java.io.InputStream;
import java.io.Serializable;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.filter.RecordFilterNotFoundException;

/**
 * The Blackboard is a container for a set of records that are processed in a single step, e.g. to add a crawled record
 * to SMILA in connectivity or to process a single queue message. The blackboard can be connected to storage services
 * (binary storage, record storage) so that existing versions of records can be loaded before the processing starts and
 * the result records can be persisted afterwards. However, a blackboard can also be just "transient", i.e. not
 * connected to any storage services. Then no data will be persisted automatically by the blackboard, the client must
 * care about handling of the results in this case.
 */
public interface Blackboard {
  /**
   * @return the data factory used by this blackboard to create records.
   */
  DataFactory getDataFactory();

  /** use as an argument in {@link #getRecord(String, CreateMode). */
  enum Get {
    /** do not create a new record if it does not exists on blackboard or in record storage. */
    EXISTING,
    /** create a new record if it does not exist in record storage. */
    AUTO_CREATE,
    /** always create a new empty record that will replace an existing version on commit. */
    NEW
  }

  /**
   * get a reference to the record on the blackboard. See documentation of the values of {@link Get} for details on the
   * mode parameter.
   * 
   * @param id
   *          the id
   * @param mode
   *          configure behavior of method.
   * @return the newly created or existing record.
   */
  Record getRecord(String id, Get mode) throws BlackboardAccessException;

  /**
   * Get current version of record with given ID. Same as {@link #getRecord(String, Get)} with {@link Get#EXISTING}.
   * This method is usually the correct one to use in pipelets or workers to process a record.
   * 
   * @param id
   *          the id
   * 
   * @return the record if it exists on blackboard or in record storage (if available), else null.
   * 
   * @throws BlackboardAccessException
   *           error accessing attached storages
   */
  Record getRecord(String id) throws BlackboardAccessException;

  /**
   * Shortcut for {@link #getRecord(String)} + {@link Record#getMetadata()}. If {@link #getRecord(String)} would yield
   * null, this method yields null, too.
   * 
   * @param id
   *          the id
   * 
   * @return the record metadata if the record exists on blackboard or in storage, else null.
   * 
   * @throws BlackboardAccessException
   *           error accessing attached storages
   */
  AnyMap getMetadata(String id) throws BlackboardAccessException;

  /**
   * Shortcut for {@link #getRecord(String)} + {@link #filterRecord(Record, String))}: Gets the blackboard record and
   * applies the named filter.
   * 
   * @param id
   *          the id
   * @param filterName
   *          the filter name
   * 
   * @return the filtered record metadata if the record exists on blackboard or in storage, else null.
   * 
   * @throws BlackboardAccessException
   *           error accessing storages for reading the record
   * 
   * @throws RecordFilterNotFoundException
   *           named record filter does not exist
   */
  Record getRecord(String id, String filterName) throws BlackboardAccessException, RecordFilterNotFoundException;

  /**
   * Filter record metadata. Attachments will not be included in the filtered record.
   * 
   * @param record
   *          the record
   * @param filterName
   *          the filter name
   * 
   * @return the record
   * 
   * @throws RecordFilterNotFoundException
   *           named record filter does not exist
   */
  Record filterRecord(final Record record, final String filterName) throws RecordFilterNotFoundException;

  /**
   * Sets a record on the blackboard. If a record with the same ID already exists, it will be replaced. If the new
   * record has attachment and a binary storage is attached to the blackboard, the attachments will be stored in binary
   * storage and removed from memory.
   * 
   * @param record
   *          the record
   * 
   * @throws BlackboardAccessException
   *           error accessing storages for reading the record or writing attachments
   */
  void setRecord(Record record) throws BlackboardAccessException;

  /**
   * Synchronize: copy all attributes and attachments from the incoming record to the blackboard record with the same
   * ID.
   * 
   * @param record
   *          the record
   * 
   * @throws BlackboardAccessException
   *           error accessing storages for reading the record or writing attachments
   */
  void synchronizeRecord(Record record) throws BlackboardAccessException;

  /**
   * remove record completely from blackboard. If storages are attached, the record data is removed from storages when
   * {@link #commit()} or {@link #commitRecord(String)} is called.
   * 
   * @param id
   *          record id
   */
  void removeRecord(String id);

  /**
   * Checks for attachment.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @return true, if the record exists and has the named attachment. Else false.
   * 
   * @throws BlackboardAccessException
   *           error accessing storages for check
   */
  boolean hasAttachment(String id, String name) throws BlackboardAccessException;

  /**
   * Gets the attachment.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @return the attachment, or null if record does not exist or does not have the attachment
   * 
   * @throws BlackboardAccessException
   *           error accessing storages for reading the record or attachment
   */
  byte[] getAttachment(String id, String name) throws BlackboardAccessException;

  /**
   * Gets the attachment as stream.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @return the attachment as stream, or null if record does not exist or does not have the attachment
   * 
   * @throws BlackboardAccessException
   *           error accessing storages for reading the record or attachment
   */
  InputStream getAttachmentAsStream(String id, String name) throws BlackboardAccessException;

  /**
   * Sets the attachment.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * @param attachment
   *          the attachment
   * 
   * @throws BlackboardAccessException
   *           error if record does not exist or on errors accessing storages for reading the record or writing the
   *           attachment
   */
  void setAttachment(String id, String name, byte[] attachment) throws BlackboardAccessException;

  /**
   * Sets the attachment from stream.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * @param attachmentStream
   *          the attachment stream
   * 
   * @throws BlackboardAccessException
   *           error if record does not exist or on errors accessing storages for reading the record or writing the
   *           attachment
   */
  void setAttachmentFromStream(String id, String name, InputStream attachmentStream)
    throws BlackboardAccessException;

  /**
   * Sets attachment from file.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * @param attachmentFile
   *          attachment file
   * 
   * @throws BlackboardAccessException
   *           error if record does not exist or on errors accessing storages for reading the record or writing the
   *           attachment
   */
  void setAttachmentFromFile(String id, String name, File attachmentFile) throws BlackboardAccessException;

  /**
   * Removes the attachment.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @throws BlackboardAccessException
   *           error if record does not exist or on errors accessing storages for reading the record or deleting the
   *           attachment
   */
  void removeAttachment(String id, String name) throws BlackboardAccessException;

  /**
   * Checks for global note.
   * 
   * @param name
   *          the name
   * 
   * @return true, if successful
   */
  boolean hasGlobalNote(String name);

  /**
   * Gets the global note.
   * 
   * @param name
   *          the name
   * 
   * @return the global note
   */
  Serializable getGlobalNote(String name);

  /**
   * Sets the global note.
   * 
   * @param name
   *          the name
   * @param object
   *          the object
   */
  void setGlobalNote(String name, Serializable object);

  /**
   * Checks for record note.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @return true, if successful
   */
  boolean hasRecordNote(String id, String name);

  /**
   * Gets the record note.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * 
   * @return the record note
   */
  Serializable getRecordNote(String id, String name);

  /**
   * Sets the record note.
   * 
   * @param id
   *          the id
   * @param name
   *          the name
   * @param object
   *          the object
   */
  void setRecordNote(String id, String name, Serializable object);

  /**
   * All changes are written to the storages before the record is removed. The record is unlocked in the database.
   * 
   * @param id
   *          the id
   * 
   * @throws BlackboardAccessException
   *           the blackboard access exception
   */
  void commitRecord(String id) throws BlackboardAccessException;

  /**
   * The record is removed from the blackboard. The record is unlocked in the database. If the record was created new
   * (not overwritten) on this blackboard it should be removed from the storage completely.
   * 
   * @param id
   *          the id
   */
  void unloadRecord(String id);

  /**
   * <p>
   * commit ALL records on this blackboard to storages (if any) and release resources.
   * </p>
   * <p>
   * The records will not be unloaded from the blackboard.
   * </p>
   * <p>
   * The operation throws an exception if at least one record could not be committed (e.g. written to storages)
   * successfully. However, the implementation should try to commit as many records as possible and not stop on the
   * first failed commit and invalidate the remaining records.
   * </p>
   * 
   * @throws BlackboardAccessException
   *           at least one record could not be committed.
   */
  void commit() throws BlackboardAccessException;

  /**
   * remove ALL records from blackboard and release all associated resources. Nothing is written to connected storage
   * services.
   */
  void unload();

}
