/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation Drazen Cindric (Attensity Europe GmbH) -
 * data model improvements
 *******************************************************************************/

package org.eclipse.smila.processing.pipelets;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.parameters.MissingParameterException;
import org.eclipse.smila.processing.parameters.ParameterAccessor;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.processing.util.ResultCollector;

/**
 * Simple pipelet that copies a String value between attributes and or attachments. It supports two modes: <li>COPY:
 * copies the value from the input to the output attribute/attachment</li> <li>MOVE: same as copy, but deletes the value
 * from the input attribute/attachment</li>
 */
public class CopyPipelet extends ATransformationPipelet {

  /**
   * property to configure the execution mode of the pipelet.
   */
  private static final String PROPERTY_MODE = "mode";

  /** The log. */
  private final Log _log = LogFactory.getLog(getClass());

  /** @return The execution mode. Default is copy. */
  protected Mode getMode(final ParameterAccessor paramAccessor) throws MissingParameterException {
    final String mode = paramAccessor.getParameter(PROPERTY_MODE, null);
    if (mode != null) {
      return Mode.valueOf(mode);
    }
    return Mode.COPY;
  }

  /** {@inheritDoc} */
  @Override
  public String[] process(final Blackboard blackboard, final String[] recordIds) throws ProcessingException {
    final ParameterAccessor paramAccessor = new ParameterAccessor(blackboard, _config);
    final ResultCollector resultCollector =
      new ResultCollector(paramAccessor, _log, ProcessingConstants.DROP_ON_ERROR_DEFAULT);
    if (recordIds != null) {
      for (final String id : recordIds) {
        try {
          paramAccessor.setCurrentRecord(id);
          final SourceType inputType = getInputType(paramAccessor);
          final SourceType outputType = getOutputType(paramAccessor);
          final String inputName = getInputName(paramAccessor);
          final String outputName = getOutputName(paramAccessor);
          // read input
          final String srcValue = readStringInput(blackboard, id, paramAccessor);
          if (srcValue != null) {
            // write to output
            storeResult(blackboard, id, srcValue, paramAccessor);
            if (_log.isTraceEnabled()) {
              _log.trace("copied value from input " + inputType + " " + inputName + " to " + outputType + " "
                + outputName);
            }
          } // if

          if (Mode.MOVE.equals(getMode(paramAccessor))) {
            // remove old value
            if (isReadFromAttribute(inputType)) {
              final AnyMap metaData = blackboard.getMetadata(id);
              metaData.remove(inputName);
            } else {
              blackboard.removeAttachment(id, inputName);
            }
            if (_log.isTraceEnabled()) {
              _log.trace("deleted value of input " + inputType + " " + inputName);
            }
          } // if
          resultCollector.addResult(id);
        } catch (final Exception ex) {
          resultCollector.addFailedResult(id, ex);
        }
      } // for
    } // if
    return resultCollector.getResultIds();
  }

  /**
   * Execution mode: copy or move value.
   */
  private enum Mode {
    /**
     * Copy the value.
     */
    COPY,

    /**
     * Move the value (deleting the value from the input source).
     */
    MOVE;
  }

}
