/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.taskmanager.Task;

/**
 * JobManager service interface.
 */
public interface JobManager {

  // ----- Jobs -----
  /**
   * @param jobName
   *          name of the job to start
   * @return id of the new job run
   * @throws JobManagerException
   *           error starting job
   */
  String startJob(String jobName) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job to start
   * @param jobMode
   *          the mode for the job to start.
   * @return id of the new job run
   * @throws JobManagerException
   *           error starting job
   */
  String startJob(String jobName, JobRunMode jobMode) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to finish the run.
   * @param jobId
   *          id of the job run to finish
   * @throws JobManagerException
   *           error finishing job
   */
  void finishJob(String jobName, String jobId) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to cancel the run.
   * @param jobId
   *          id of the job run to cancel
   * @throws JobManagerException
   *           error canceling job
   */
  void cancelJob(String jobName, String jobId) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to monitor the run.
   * @param jobId
   *          id of the job run to get the monitoring data. May be either the current run or a completed run.
   * @return monitoring data for the job run with the given id.
   * @throws JobManagerException
   *           error getting job run data
   */
  AnyMap getJobRunData(String jobName, String jobId) throws JobManagerException;

  /**
   * @param jobName
   *          name of the job for which to monitor the run.
   * @param jobId
   *          id of the job run to get the monitoring data. May be either the current run or a completed run.
   * @param returnDetails
   *          whether to return details (e.g. workflow/bucket definition)
   * @return monitoring data for the job run with the given id.
   * @throws JobManagerException
   *           error getting job run data
   */
  AnyMap getJobRunData(String jobName, String jobId, boolean returnDetails) throws JobManagerException;

  /**
   * get information about a workflow run.
   * 
   * @return a map with {@link JobManagerConstants#DATA_WORKFLOW_RUN_NO_OF_ACTIVE_TASKS} and
   *         {@link JobManagerConstants#DATA_WORKFLOW_RUN_NO_OF_TRANSIENT_BULKS}. Additional keys may be added.
   * 
   * @throws JobManagerException
   *           jobrun or workflowrun are not active anymore.
   */
  AnyMap getWorkflowRunData(String jobName, String jobId, String workflowRunId) throws JobManagerException;

  /**
   * get info on current job run.
   * 
   * @param jobName
   *          job name
   * @return if job is defined and active: id and state of current run. Else null.
   * @throws JobManagerException
   *           error getting state.
   */
  JobRunInfo getJobRunInfo(String jobName) throws JobManagerException;

  /**
   * get Ids of complete job runs.
   * 
   * @param jobName
   *          job name
   * @return Ids of completed job runs, sorted lexicographically ascending. Empty set if job is not defined or no
   *         completed job runs exists.
   * @throws JobManagerException
   *           error getting state.
   */
  Collection<String> getCompletedJobRunIds(String jobName) throws JobManagerException;

  /**
   * delete a previously completed job run.
   * 
   * @param jobName
   *          name of the job for which to monitor the run. 
   * @param jobId
   *          id of the complete job run to delete the monitoring data.
   * @throws JobManagerException
   *           run id is not a completed run or error deleting it.
   */
  void deleteJobRunData(String jobName, String jobId) throws JobManagerException;

  /**
   * @return the DefinitionPersistence reference.
   */
  DefinitionPersistence getDefinitionPersistence();

  /**
   * @param worker
   *          The worker
   * @return The matching task generator
   */
  TaskGenerator getTaskGenerator(final WorkerDefinition worker);

  /**
   * Finishes a task and retrieves the following tasks. Dependent on the resultDescription job manager can decide what
   * to do.
   * 
   * @param currentTask
   *          The finished task.
   * @return A list of follow up tasks. If there are no follow-up tasks, an empty list will be generated.
   * @throws JobManagerException
   *           error while trying to finish the task.
   */
  List<Task> finishTask(Task currentTask) throws JobManagerException;

  /**
   * Creates an initial task for input-workers.
   * 
   * @param workerName
   *          The name of the worker for which this task is requested.
   * @param id
   *          The id of the task-source (e.g. index name, job id, ...) for which the task is requested. The TaskProducer
   *          and the worker requester of the tasks should have a common understanding of what this id is about.
   * @return A task for the worker or null if task delivery is currently not allowed for this id.
   * @throws JobManagerException
   *           error while trying to generate initial task. The job has not the correct state or the worker is not an
   *           startAction worker.
   */
  Task getInitialTask(String workerName, String id) throws JobManagerException;

  /**
   * Creates a map with job runs using the given store.
   * 
   * @param storeName
   *          the store name
   * @return a map with all job runs using the store with given name. The job name is used as key, the job run id as
   *         value.
   */
  Map<String, String> getJobRunsUsingStore(String storeName);
}
