/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.jobmanager.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.persistence.ConfigNotFoundException;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.osgi.service.component.ComponentContext;

/**
 * Implements the handling of HTTP requests to retrieve a data object type definition. <br>
 * 
 * URL pattern: <code>smila/jobmanager/dataobjecttypes/-dotname-/</code> <br>
 * 
 * Methods allowed: <code>GET</code>
 */
public class DataObjectTypeHandler extends JsonRequestHandler {

  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void activate(final ComponentContext context) {
    super.activate(context);
    _defPersistence = _jobManager.getDefinitionPersistence();
  }

  /**
   * clean up JobManager on service shutdown.
   * 
   * @param context
   *          OSGi service context.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final String name = getName(requestUri);
    boolean returnDetails = false;
    if (inputRecord != null && inputRecord.getMetadata().containsKey(JobManagerConstants.KEY_RETURN_DETAILS)) {
      returnDetails =
        inputRecord.getMetadata().getBooleanValue(JobManagerConstants.KEY_RETURN_DETAILS).booleanValue();
    }
    final DataObjectTypeDefinition dot = _defPersistence.getDataObjectType(name);
    if (dot == null) {
      throw new ConfigNotFoundException("Data object type '" + name + "' not found.");
    }
    return dot.toAny(returnDetails);
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * Gets the DataObjectType name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he DataObjectType name from the requestUri.
   */
  private String getName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }
}
