/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.WorkerDefinition;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.jobmanager.persistence.PersistenceException;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;
import org.osgi.service.component.ComponentContext;

/**
 * Implements the handling of HTTP requests to request the worker list. <br>
 * 
 * URL pattern: <code>smila/jobmanager/workers/</code> <br>
 * 
 * Methods allowed: <code>GET</code>
 */
public class WorkersHandler extends JsonRequestHandler {
  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void activate(final ComponentContext context) {
    super.activate(context);
    _defPersistence = _jobManager.getDefinitionPersistence();
  }

  /**
   * clean up JobManager on service shutdown.
   * 
   * @param context
   *          OSGi service context.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final Map<String, Object> resultMap = new HashMap<String, Object>();
    final List<Map<String, Object>> workerList = new ArrayList<Map<String, Object>>();
    for (final String workerDefinitionName : _defPersistence.getWorkers()) {
      final LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>();
      map.put("name", workerDefinitionName);
      map.put("url", URLCreator.create(getRequestHost(), requestUri, workerDefinitionName));
      workerList.add(map);
    }
    resultMap.put(WorkerDefinition.KEY_WORKERS, workerList);
    return resultMap;
  }

  /**
   * Adds HTTP result code 500 (INTERNAL SERVER ERROR) for PersistenceExceptions to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof NotFoundHTTPResult) {
      // NotFoundHTTPResult before PersistenceException! (Because of ConfigNotFoundException.)
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof PersistenceException) {
      return HttpStatus.INTERNAL_SERVER_ERROR;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }
}
