/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.List;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerException;

/**
 * Implements the handling of HTTP requests to get workflow run information.
 * 
 * URL pattern: smila/jobmanager/jobs/-jobname-/-jobid-/workflowrun/-workflowrunid-
 */
public class WorkflowRunHandler extends JsonRequestHandler {

  /** valid number of dynamic URI parts. */
  private static final int URIPARTS_COUNT = 3;

  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() == URIPARTS_COUNT) {
      final String jobName = uriParts.get(0);
      final String jobId = uriParts.get(1);
      final String workflowRunId = uriParts.get(2);
      return _jobManager.getWorkflowRunData(jobName, jobId, workflowRunId);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'");
  }

  /**
   * 
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof JobManagerException) {
      return HttpStatus.NOT_FOUND;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * {@inheritDoc}
   * 
   * GET is currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }

}
