/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.persistence;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.ipc.IpcSerializationUtils;
import org.eclipse.smila.jobmanager.BucketDefinition;
import org.eclipse.smila.jobmanager.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.IllegalJobStateException;
import org.eclipse.smila.jobmanager.InvalidConfigException;
import org.eclipse.smila.jobmanager.JobDefinition;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.JobManagerException;
import org.eclipse.smila.jobmanager.WorkerDefinition;
import org.eclipse.smila.jobmanager.WorkflowDefinition;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.utils.config.ConfigUtils;
import org.eclipse.smila.utils.config.ConfigurationLoadException;

/**
 * Handles read/write operations of jobmanager definitions.
 * 
 * Write operations are handled directly by writing to internal store after validating them. Read operations are
 * delegated to DefinitionAccessor.
 * 
 */
public class DefinitionPersistence {

  /** Configuration file with predefined buckets. */
  private static final String FILENAME_PREDEFINED_BUCKETS = "buckets.json";

  /** Configuration file with predefined data object types. */
  private static final String FILENAME_PREDEFINED_DATA_OBJECT_TYPES = "dataObjectTypes.json";

  /** Configuration file with predefined workers. */
  private static final String FILENAME_PREDEFINED_WORKERS = "workers.json";

  /** Configuration file with predefined workflows. */
  private static final String FILENAME_PREDEFINED_WORKFLOWS = "workflows.json";

  /** definition storage used for write operations. */
  private final DefinitionStorage _storage;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** for record<->BON/JSON serialization. */
  private final IpcSerializationUtils _serializationUtils = new IpcSerializationUtils();

  /** for reading definitions from storage or configuration. */
  private final DefinitionAccessor _definitionAccessor;

  /** for validating definitions before writing them to storage. */
  private final DefinitionValidator _definitionValidator;

  /**
   * Create object with given underlying definition storage service. The predefined data object type definitions,
   * buckets and worker definitions are read from the configuration area.
   * 
   * @param storage
   *          definition storage for persistence of user-submitted definitions.
   * @param jobManager
   *          the reference to the jobmanager service
   * @param objectStoreService
   *          a reference to the object store service.
   */
  public DefinitionPersistence(final DefinitionStorage storage, final JobManager jobManager,
    final ObjectStoreService objectStoreService) {
    if (storage == null) {
      throw new IllegalArgumentException("DefinitionStorage must not be <null>.");
    }
    if (jobManager == null) {
      throw new IllegalArgumentException("JobManager must not be <null>.");
    }
    _definitionAccessor = new DefinitionAccessor();
    _definitionValidator = new DefinitionValidator(_definitionAccessor, jobManager, objectStoreService);

    readConfiguredDataObjectTypeDefinitions();
    readConfiguredBucketDefinitions();
    readConfiguredWorkerDefinitions();
    readConfiguredWorkflowDefinitions();

    _storage = storage;
    _definitionAccessor.setStorage(storage);
  }

  /**
   * @return all data object type names.
   */
  public Collection<String> getDataObjectTypes() {
    return _definitionAccessor.getDataObjectTypes();
  }

  /**
   * @param name
   *          the name of the data object type definition
   * @return the data object type definition for the given name, or <code>null</code> if not found
   */
  public DataObjectTypeDefinition getDataObjectType(final String name) {
    return _definitionAccessor.getDataObjectType(name);
  }

  /**
   * @return all worker names.
   */
  public Collection<String> getWorkers() {
    return _definitionAccessor.getWorkers();
  }

  /**
   * @param name
   *          the name of the worker definition
   * @return returns the worker definition with the given name, or <code>null</code> if not found
   */
  public WorkerDefinition getWorker(final String name) {
    return _definitionAccessor.getWorker(name);
  }

  /**
   * @return names of preconfigured and user-defined buckets.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public Collection<String> getBuckets() throws PersistenceException {
    return _definitionAccessor.getBuckets();
  }

  /**
   * get a bucket from configuration or storage.
   * 
   * @param name
   *          bucket name
   * @return bucket definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public BucketDefinition getBucket(final String name) throws PersistenceException {
    return _definitionAccessor.getBucket(name);
  }

  /**
   * add one bucket to definition storage. No further validation.
   * 
   * @param bucketDefinition
   *          new bucket.
   * @throws PersistenceException
   *           error writing to storage.
   * @throws InvalidConfigException
   *           error adding definition due to dependency constraints.
   */
  public void addBucket(final BucketDefinition bucketDefinition) throws PersistenceException,
    InvalidConfigException {
    _definitionValidator.validateBucket(bucketDefinition);
    _definitionValidator.validateBucketWorkflows(bucketDefinition);
    _storage.addBucket(bucketDefinition);
  }

  /**
   * remove a bucket from the storage.
   * 
   * @param name
   *          bucket name.
   * @throws PersistenceException
   *           error deleting definition.
   * @throws InvalidConfigException
   *           error deleting definition due to dependency constraints
   */
  public void removeBucket(final String name) throws PersistenceException, InvalidConfigException {
    _definitionValidator.checkRemoveBucket(name);
    _storage.removeBucket(name);
  }

  /**
   * @return names of defined jobs.
   * @throws PersistenceException
   *           error reading from storage
   */
  public Collection<String> getJobs() throws PersistenceException {
    final List<String> jobNames = new ArrayList<String>();
    if (_storage != null) {
      final Collection<String> storageJobs = _storage.getJobs();
      if (storageJobs != null) {
        jobNames.addAll(storageJobs);
      }
    }
    return jobNames;
  }

  /**
   * get a job name from configuration or storage.
   * 
   * @param name
   *          job name
   * @return job definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public JobDefinition getJob(final String name) throws PersistenceException {
    if (_storage != null) {
      return _storage.getJob(name);
    }
    return null;
  }

  /**
   * @param jobName
   *          job name
   * @return true if a job with this name is defined
   * @throws PersistenceException
   */
  public boolean hasJob(final String jobName) throws PersistenceException {
    if (_storage != null) {
      return _storage.hasJob(jobName);
    }
    return false;
  }

  /**
   * remove a job from the storage.
   * 
   * @param name
   *          job name.
   * @throws PersistenceException
   *           error deleting definition
   * @throws IllegalJobStateException
   *           the job is in an illegal state (i.e. it is running and cannot be removed).
   */
  public void removeJob(final String name) throws PersistenceException, IllegalJobStateException {
    try {
      _definitionValidator.checkJobRemove(name);
    } catch (final IllegalJobStateException e) {
      throw e;
    } catch (final JobManagerException e) {
      throw new PersistenceException("Exception while checking job '" + name + "' for deletion.", e);
    }
    _storage.removeJob(name);
  }

  /**
   * add a job to definition storage. No further validation.
   * 
   * @param jobDefinition
   *          new job.
   * @throws PersistenceException
   *           error writing to storage.
   * @throws InvalidConfigException
   *           the job is misconfigured (e.g. has invalid parameter settings or not enough parameters to satisfy data
   *           object type definitions or worker parameters).
   */
  public void addJob(final JobDefinition jobDefinition) throws PersistenceException, InvalidConfigException {
    _definitionValidator.validateJob(jobDefinition);
    _storage.addJob(jobDefinition);
  }

  /**
   * @return (unique) names of defined workflows.
   * @throws PersistenceException
   *           error reading from storage
   */
  public Collection<String> getWorkflows() throws PersistenceException {
    return _definitionAccessor.getWorkflows();
  }

  /**
   * get a workflow definition from storage (try first) or configuration (if not found in storage).
   * 
   * @param name
   *          workflow name
   * @return workflow definition.
   * @throws PersistenceException
   *           error reading from storage.
   */
  public WorkflowDefinition getWorkflow(final String name) throws PersistenceException {
    return _definitionAccessor.getWorkflow(name);
  }

  /**
   * remove a workflow from the storage.
   * 
   * @param name
   *          workflow name.
   * @throws PersistenceException
   *           error deleting definition
   * @throws InvalidConfigException
   *           error deleting definition due to dependency constraints
   */
  public void removeWorkflow(final String name) throws PersistenceException, InvalidConfigException {
    _definitionValidator.checkRemoveWorkflow(name);
    _storage.removeWorkflow(name);
  }

  /**
   * add a workflow to definition storage. No further validation.
   * 
   * @param workflowDefinition
   *          new workflow.
   * @throws PersistenceException
   *           error writing to storage.
   * @throws InvalidConfigException
   *           the configuration of the workflow is incorrect (e.g. would lead to an invalid store name.)
   */
  public void addWorkflow(final WorkflowDefinition workflowDefinition) throws PersistenceException,
    InvalidConfigException {
    _definitionValidator.validateWorkflow(workflowDefinition);
    _storage.addWorkflow(workflowDefinition);
  }

  /**
   * read predefined buckets from configuration area. Tries to load as many definitions as possible, invalid definitions
   * are ignored, see log for details.
   */
  private void readConfiguredBucketDefinitions() {
    try {
      final InputStream bucketsJson =
        ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_BUCKETS);
      try {
        final AnyMap bucketsAny = (AnyMap) _serializationUtils.jsonStream2any(bucketsJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<BucketDefinition> buckets = BucketDefinition.parseBuckets(bucketsAny, true);
        final Map<String, BucketDefinition> configuredBuckets = new HashMap<String, BucketDefinition>();
        for (final BucketDefinition bucket : buckets) {
          if (configuredBuckets.containsKey(bucket.getName())) {
            _log.warn("Configuration contains duplicate bucket definition '" + bucket.getName()
              + "', ignoring second one.");
          } else {
            try {
              _definitionValidator.validateBucket(bucket);
              configuredBuckets.put(bucket.getName(), bucket);
            } catch (final InvalidConfigException ex) {
              _log.warn("Configuration contains invalid bucket definition for bucket '" + bucket.getName() + "'",
                ex);
            }
          }
        }
        _definitionAccessor.setConfiguredBuckets(configuredBuckets);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined buckets from configuration area, no predefined buckets available.", ex);
      } catch (final PersistenceException ex) {
        _log.warn("Error reading predefined buckets from configuration area, no predefined buckets available.", ex);
      } catch (final InvalidConfigException e) {
        _log.warn("Error parsing predefined bucket definitions from configuration area, "
          + "no predefined bucket definitions available.", e);
      } finally {
        IOUtils.closeQuietly(bucketsJson);
      }
    } catch (final ConfigurationLoadException ex) {
      _log.info("No predefined buckets found in configuration area.");
    }
  }

  /**
   * Reads the predefined data object type definitions from configuration area.
   */
  private void readConfiguredDataObjectTypeDefinitions() {
    try {
      final InputStream dotJson =
        ConfigUtils
          .getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_DATA_OBJECT_TYPES);
      try {
        final AnyMap dotMap = (AnyMap) _serializationUtils.jsonStream2any(dotJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<DataObjectTypeDefinition> dots =
          DataObjectTypeDefinition.parseDataObjectTypeDefinitions(dotMap, true);
        final Map<String, DataObjectTypeDefinition> configuredDataObjectTypes =
          new HashMap<String, DataObjectTypeDefinition>();
        for (final DataObjectTypeDefinition dot : dots) {
          configuredDataObjectTypes.put(dot.getName(), dot);
        }
        _definitionAccessor.setConfiguredDataObjectTypes(configuredDataObjectTypes);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined data object type definitions from configuration area, "
          + "no predefined data object type definitions available.", ex);
      } catch (final InvalidConfigException e) {
        _log.warn("Error parsing predefined data object type definitions from configuration area, "
          + "no predefined worker definitions available.", e);
      } finally {
        IOUtils.closeQuietly(dotJson);
      }
    } catch (final ConfigurationLoadException cle) {
      _log.info("No predefined data object types found in configuration area.");
    }
  }

  /**
   * Reads the predefined worker definitions from configuration area.
   */
  private void readConfiguredWorkerDefinitions() {
    try {
      final InputStream workerJson =
        ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_WORKERS);
      try {
        final AnyMap workerMap = (AnyMap) _serializationUtils.jsonStream2any(workerJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<WorkerDefinition> workers = WorkerDefinition.parseWorkerDefinitions(workerMap, true);
        final Map<String, WorkerDefinition> configuredWorkers = new HashMap<String, WorkerDefinition>();
        for (final WorkerDefinition worker : workers) {
          configuredWorkers.put(worker.getName(), worker);
        }
        _definitionAccessor.setConfiguredWorkers(configuredWorkers);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined worker definitions from configuration area, "
          + "no predefined worker definitions available.", ex);
      } catch (final InvalidConfigException e) {
        _log.warn("Error parsing predefined worker definitions from configuration area, "
          + "no predefined worker definitions available.", e);
      } finally {
        IOUtils.closeQuietly(workerJson);
      }
    } catch (final ConfigurationLoadException cle) {
      _log.info("No predefined workers found in configuration area.");
    }
  }

  /**
   * read predefined workflows from configuration area. Tries to load as many definitions as possible, invalid
   * definitions are ignored, see log for details.
   */
  private void readConfiguredWorkflowDefinitions() {
    try {
      final InputStream workflowsJson =
        ConfigUtils.getConfigStream(JobManagerConstants.CONFIGURATION_BUNDLE, FILENAME_PREDEFINED_WORKFLOWS);
      try {
        final AnyMap workflowsAnyMap = (AnyMap) _serializationUtils.jsonStream2any(workflowsJson);
        // add a readonly flag for the preconfigured definitions.
        final Collection<WorkflowDefinition> workflows = WorkflowDefinition.parseWorkflows(workflowsAnyMap, true);
        final Map<String, WorkflowDefinition> configuredWorkflows = new HashMap<String, WorkflowDefinition>();
        for (final WorkflowDefinition workflow : workflows) {
          if (configuredWorkflows.containsKey(workflow.getName())) {
            _log.warn("Configuration contains duplicate workflow definition '" + workflow.getName()
              + "', ignoring second one.");
          } else {
            try {
              _definitionValidator.validateWorkflow(workflow);
              configuredWorkflows.put(workflow.getName(), workflow);
            } catch (final InvalidConfigException ex) {
              _log.warn("Configuration contains invalid workflow definition for workflow '" + workflow.getName()
                + "'.", ex);
            } catch (final PersistenceException e) {
              _log.warn("Error while trying to validate workflow '" + workflow.getName() + "'.", e);
            }
          }
        }
        _definitionAccessor.setConfiguredWorkflows(configuredWorkflows);
      } catch (final IOException ex) {
        _log.warn("Error reading predefined workflows from configuration area, no predefined workflows available.",
          ex);
      } catch (final InvalidConfigException e) {
        _log.warn("Error parsing predefined workflows from configuration area, "
          + "no predefined workflows definitions available.", e);
      } finally {
        IOUtils.closeQuietly(workflowsJson);
      }
    } catch (final ConfigurationLoadException ex) {
      _log.info("No predefined workflows found in configuration area.");
    }
  }

}
