/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;

/**
 * Manager of index update operations (add, delete, update records).
 * 
 */
public interface BulkbuilderService {

  /**
   * add a record to a bulk to insert it to the given job. Create a new bulk if none is open. Finish it and create a
   * task for it if the maximum bulk size is exceeded. If record is null an IllegalArgumentException is thrown.
   * 
   * @param jobName
   *          job name
   * @param record
   *          record to add.
   * @return Id of job and workflow run to which the records was added.
   * @throws BulkbuilderException
   *           error writing the record.
   * @throws InvalidRecordException
   *           record does not conform to schema.
   */
  WorkflowRunInfo addRecord(final String jobName, final Record record) throws BulkbuilderException,
    InvalidRecordException;

  /**
   * add a record to a bulk to delete it from the given job. Create a new bulk if none is open. Finish it and create a
   * task for it if the maximum bulk size is exceeded. If record is null an IllegalArgumentException is thrown.
   * 
   * @param jobName
   *          job name
   * @param record
   *          record to add.
   * @return Id of job and workflow run to which the records was added.
   * @throws BulkbuilderException
   *           error writing the record.
   * @throws InvalidRecordException
   *           record does not conform to schema.
   */
  WorkflowRunInfo deleteRecord(final String jobName, final Record record) throws BulkbuilderException,
    InvalidRecordException;

  /**
   * commit the job with the given job name.
   * 
   * @param jobName
   *          job name
   * @return Id of job and workflow run to which the records was added.
   * @throws BulkbuilderException
   *           error committing the job
   */
  WorkflowRunInfo commitJob(final String jobName) throws BulkbuilderException;

  /**
   * Adds a record to a micro bulk with the given id.
   * 
   * @param jobName
   *          the job name
   * @param record
   *          The record
   * @param microBulkId
   *          The id of the micro bulk
   * @return The workflow run info.
   * @throws BulkbuilderException
   *           error writing the record.
   * @throws InvalidRecordException
   *           record does not conform to schema.
   */
  WorkflowRunInfo addToMicroBulk(final String jobName, final Record record, final String microBulkId)
    throws BulkbuilderException, InvalidRecordException;

  /**
   * Finishes the micro bulk with the given id. This also removes the micro bulk after successful processing.
   * 
   * @param jobName
   *          the job name
   * @param microBulkId
   *          The id
   * @return the workflow run info
   * @throws BulkbuilderException
   *           error writing the micro bulk.
   */
  WorkflowRunInfo finishMicroBulk(final String jobName, final String microBulkId) throws BulkbuilderException;

  /**
   * Remove the micro bulk with the given id.
   * 
   * @param microBulkId
   *          The micro bulk to remove
   */
  void removeMicroBulk(final String microBulkId);
}
