/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.connectivity.framework.util;

import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.utils.digest.DigestHelper;

/**
 * A factory to create Hash objects in Crawlers and Agents.
 */
public final class ConnectivityHashFactory {

  /**
   * singleton instance.
   */
  private static ConnectivityHashFactory s_instance;

  /**
   * Default Constructor.
   */
  private ConnectivityHashFactory() {
  }

  /**
   * Returns the singleton instance of the HashFactory.
   * 
   * @return the HashFactory
   */
  public static ConnectivityHashFactory getInstance() {
    if (s_instance == null) {
      s_instance = new ConnectivityHashFactory();
    }
    return s_instance;
  }

  /**
   * Create a hash object based on the given Attributes.
   * <p/>
   * <strong>Note</strong> the order of elementds in the passed AnyMaps is important and reflected in the generated hash
   * and id. Usually this is not wanted but for performance reasons it is better to take the order as is and ensure
   * proper ordering in the crawler/agent.
   * 
   * @param hashAttributesMap
   *          an AnyMap of attributes whose values are used to create the hash. Must not be null or empty.
   * @return a String containing the hash
   */
  public String createHash(final AnyMap hashAttributesMap) {
    if (hashAttributesMap == null || hashAttributesMap.size() == 0) {
      throw new IllegalArgumentException("Parameter hashAttributes must not be null or empty");
    }
    return createHash(hashAttributesMap, null);
  }

  /**
   * Create a hash object based on the given Attachments.
   * <p/>
   * <strong>Note</strong> the order of elementds in the passed AnyMaps is important and reflected in the generated hash
   * and id. Usually this is not wanted but for performance reasons it is better to take the order as is and ensure
   * proper ordering in the crawler/agent.
   * 
   * @param hashAttachments
   *          a Map of attachment names and attachment String values, that are used to create the hash. Must not be null
   *          or empty.
   * @return a String containing the hash
   */
  public String createHash(final Map<String, ?> hashAttachments) {
    if (hashAttachments == null || hashAttachments.isEmpty()) {
      throw new IllegalArgumentException("Parameter hashAttachments must not be null or empty");
    }
    return createHash(null, hashAttachments);
  }

  /**
   * Create a hash object based on the given Attributes and Attachments. One of the parameters may be null or empty.
   * <p/>
   * <strong>Note</strong> the order of elementds in the passed AnyMaps is important and reflected in the generated hash
   * and id. Usually this is not wanted but for performance reasons it is better to take the order as is and ensure
   * proper ordering in the crawler/agent.
   * 
   * @param hashAttributeMap
   *          an AnyMap of attributes whose values are used to create the hash
   * @param hashAttachments
   *          a Map of attachment names and attachment String values, that are used to create the hash.
   * @return a String containing the hash
   */
  public String createHash(final AnyMap hashAttributeMap, final Map<String, ?> hashAttachments) {
    if ((hashAttributeMap == null || hashAttributeMap.size() == 0)
      && (hashAttachments == null || hashAttachments.isEmpty())) {
      throw new IllegalArgumentException(
        "Parameters hashAttributes and hashAttachments must not both be null or empty");
    }
    final StringBuilder buffer = new StringBuilder();
    appendAttributeValues(hashAttributeMap, buffer);
    appendAttachmentValues(hashAttachments, buffer);
    return DigestHelper.calculateDigest(buffer.toString());
  }

  /** append attribute names and values to buffer for creating a hash value. */
  private void appendAttributeValues(final AnyMap hashAttributeMap, final StringBuilder buffer) {
    if (hashAttributeMap != null) {
      for (final Entry<String, Any> entry : hashAttributeMap.entrySet()) {
        final String key = entry.getKey();
        final Any value = entry.getValue();
        if (value == null) {
          throw new IllegalArgumentException("hashAttributeMap entry has null value! attribute: " + key);
        }
        final String string = value.toString();
        buffer.append(key).append('=').append(string).append(';');
      }
    }
  }

  /** append attachment name and values to buffer for creating a hash value. */
  private void appendAttachmentValues(final Map<String, ?> hashAttachments, final StringBuilder buffer) {
    if (hashAttachments != null) {
      for (final Entry<String, ?> hashAttachmentEntry : hashAttachments.entrySet()) {
        final String attachmentName = hashAttachmentEntry.getKey();
        final Object attachmentValue = hashAttachmentEntry.getValue();
        if (attachmentValue == null) {
          throw new IllegalArgumentException("hashAttachmentsMap entry has null value! attachment: "
            + attachmentName);
        }
        if (attachmentValue instanceof String) {
          buffer.append(attachmentName).append('=').append(attachmentValue).append(';');
        } else if (attachmentValue instanceof byte[]) {
          buffer.append(attachmentName).append('=').append(DigestHelper.calculateDigest((byte[]) attachmentValue))
            .append(';');
        } else {
          throw new IllegalArgumentException("Attachments must be of type String or byte[]");
        }
      }
    }
  }

  /**
   * Computes a hash for the value.
   * 
   * @param value
   *          the value to compute the hash for.
   * @return the computed hash.
   */
  public String createHash(final String value) {
    return DigestHelper.calculateDigest(value);
  }
}
