/***********************************************************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Peter Wissel (brox IT Solutions GmbH) - initial API and implementation
 **********************************************************************************************************************/

package org.eclipse.smila.solr.search;

import static org.apache.commons.lang.StringUtils.isBlank;

import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.FacetParams;
import org.apache.solr.common.params.ShardParams;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.search.api.QueryConstants;
import org.eclipse.smila.solr.SolrConstants;

/**
 * The SolrQueryConverter class.
 * 
 * @author tmenzel
 * 
 */
public class SolrQueryConverter {

  /**
   * The SolrQueryParameterAccessor.
   */
  private final SolrQueryParameterAccessor _accessor;

  /**
   * The SolrQuery.
   */
  private final SolrQuery _solrQuery = new SolrQuery();

  /**
   * Constructor.
   * 
   * @param accessor
   *          the solr query parameter accessor.
   */
  public SolrQueryConverter(SolrQueryParameterAccessor accessor) {
    _accessor = accessor;
  }

  /**
   * Convert the record to a solr query string.
   * 
   * @return the SolrQuery.
   */
  public SolrQuery toSolrQuery() {
    final String qt = _accessor.getRequestHandler();
    _solrQuery.setQueryType(qt);

    doQuerySettings();
    doTermsSettings();
    doFilterSettings();
    doFacetSettings();
    doHighlightingSettings();
    doShardsSettings();
    doSpellCheckSettings();

    return _solrQuery;
  }

  /**
   * Do query settings.
   */
  private void doQuerySettings() {
    // Query
    final String q = _accessor.getQuery();
    _solrQuery.setQuery(q);
    // Paging
    final int start = _accessor.getOffset();
    _solrQuery.setStart(start);
    final int rows = _accessor.getMaxCount();
    _solrQuery.setRows(rows);
    // Field list
    final String[] fl = _accessor.getResultAttributes().toArray(new String[_accessor.getResultAttributes().size()]);
    _solrQuery.setFields(fl);
    // TODO: setIncludeScore is evidently not working, add score field manually.
    // _solrQuery.setIncludeScore(true);
    // must always have these or else building the result will fail
    _solrQuery.addField(SolrConstants.CORE_FIELD_SCORE);
    _solrQuery.addField(SolrConstants.CORE_FIELD_ID);

  }

  /**
   * Do filter settings.
   */
  private void doFilterSettings() {
    final AnySeq seq = _accessor.getFilterQuery();
    if (seq != null) {
      final String[] fq = seq.asStrings().toArray(new String[seq.size()]);
      _solrQuery.setFilterQueries(fq);
    }
  }

  /**
   * Do facet settings.
   */
  private void doFacetSettings() {
    for (AnyMap groupby : _accessor.getGroupByConfig()) {
      final String attribute = groupby.getStringValue(QueryConstants.ATTRIBUTE);
      if (attribute.equals(SolrConstants.GLOBAL)) {
        addFacetParameter(groupby, null);
      } else {
        final String facet = groupby.getStringValue(SolrConstants.FACET_ATTR);
        if (!facet.equals(FacetParams.FACET_QUERY)) {
          _solrQuery.add(facet, attribute);
        }
        addFacetParameter(groupby, attribute);
      }
    }
  }

  /**
   * Add parameter to facet attribute.
   * 
   * @param groupby
   *          the groupby configuration map.
   * @param field
   *          if not blank then the config is added for the given field name (field level config) otherwise global
   */
  private void addFacetParameter(AnyMap groupby, String field) {
    for (String key : groupby.keySet()) {
      if (key.equals(QueryConstants.ATTRIBUTE)) {
        continue;
      } else if (key.equals(SolrConstants.FACET_ATTR)) {
        continue;
      } else if (key.equals(SolrConstants.FACET_QUERY_ATTR)) {
        for (String fc : groupby.getSeq(key).asStrings()) {
          _solrQuery.add(FacetParams.FACET_QUERY, field + ":" + fc);
        }
        continue;
      } else {
        String prefix = "";
        if (StringUtils.isNotBlank(field)) {
          prefix = SolrConstants.FIELD_PREFIX + field + SolrConstants.FIELD_SUFFIX;
        }
        _solrQuery.add(prefix + key, groupby.getStringValue(key));
      }
    }
  }

  /**
   * Do terms settings.
   */
  private void doTermsSettings() {
    final AnyMap terms = _accessor.getTerms();
    if (terms != null) {
      _solrQuery.setParam(CommonParams.QT, "/terms");
      setParameter(terms, null);
    }
  }

  /**
   * Do highlighting settings.
   */
  private void doHighlightingSettings() {
    final AnySeq seq = _accessor.getHighlighting();
    if (seq != null) {
      for (Any map : seq) {
        if (map.isMap()) {
          final AnyMap highlighting = map.asMap();
          final String attribute = highlighting.getStringValue(QueryConstants.ATTRIBUTE);
          if (attribute.equals(SolrConstants.GLOBAL)) {
            setParameter(highlighting, null);
          } else {
            setParameter(highlighting, attribute);
          }
        }
      }
    }
  }

  /**
   * Set query parameter.
   * 
   * @param map
   *          the map containing parameter as key value pairs.
   * @param field
   *          if not blank then the config is added for the given field name (field level config) otherwise global
   */
  private void setParameter(AnyMap map, String field) {
    for (Entry<String, Any> entry : map.entrySet()) {

      String key = entry.getKey();
      if (key.equals(QueryConstants.ATTRIBUTE)) {
        continue;
      }
      final String value = entry.getValue().asValue().asString();
      if (isBlank(field)) {
        _solrQuery.setParam(key, value);
      } else {
        final String fieldKey = SolrConstants.FIELD_PREFIX + field + SolrConstants.FIELD_SUFFIX + key;
        _solrQuery.setParam(fieldKey, value);
      }
    }
  }

  /**
   * Do shards settings.
   */
  private void doShardsSettings() {
    final AnySeq seq = _accessor.getShards();
    if (seq != null) {
      final String shards = StringUtils.join(seq.asStrings(), ",");
      _solrQuery.setParam(ShardParams.SHARDS, shards);
    }
  }

  /**
   * Do spellcheck settings.
   */
  private void doSpellCheckSettings() {
    final AnyMap spellcheck = _accessor.getSpellcheck();
    if (spellcheck != null) {
      setParameter(spellcheck, null);
    }
  }

}
