/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.BucketDefinition;
import org.eclipse.smila.jobmanager.DefinitionBase;
import org.eclipse.smila.jobmanager.InvalidConfigException;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.processing.ProcessingException;
import org.osgi.service.component.ComponentContext;

/**
 * Implements the handling of HTTP requests to request the buckets list or to define new buckets. <br>
 * 
 * URL pattern: <code>smila/jobmanager/buckets/</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class BucketsHandler extends JsonRequestHandler {

  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void activate(final ComponentContext context) {
    super.activate(context);
    _defPersistence = _jobManager.getDefinitionPersistence();
  }

  /**
   * clean up JobManager on service shutdown.
   * 
   * @param context
   *          OSGi service context.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      final Map<String, Object> resultMap = new HashMap<String, Object>();
      final List<Map<String, Object>> bucketList = new ArrayList<Map<String, Object>>();
      for (final String bucketName : _defPersistence.getBuckets()) {
        final LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>();
        map.put("name", bucketName);
        map.put("url", URLCreator.create(getRequestHost(), requestUri, bucketName));
        bucketList.add(map);
      }
      resultMap.put(BucketDefinition.KEY_BUCKETS, bucketList);
      return resultMap;
    } else { // POST Only GET and POST are allowed, see code below
      if (inputRecord == null) {
        throw new ProcessingException("Missing input, cannot create buckets from nothing.");
      }
      final AnyMap definition = inputRecord.getMetadata();
      // create timestamp for this workflow and store it with the data
      final Any timestamp = inputRecord.getFactory().createDateTimeValue(new Date());
      definition.put(DefinitionBase.KEY_TIMESTAMP, timestamp);
      // remove readOnly flag if present.
      definition.remove(DefinitionBase.KEY_READ_ONLY);
      final BucketDefinition createdBucket = BucketDefinition.parseBucket(definition);
      _defPersistence.addBucket(createdBucket);
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      result.put("name", createdBucket.getName());
      result.put(JobManagerConstants.TIMESTAMP, timestamp);
      result.put("url", URLCreator.create(getRequestHost(), requestUri, createdBucket.getName()).toExternalForm());
      return result;
    }
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for InvalidConfigException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof InvalidConfigException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if ("POST".equals(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid method.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }
}
