/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.Bucket;
import org.eclipse.smila.jobmanager.BucketDefinition;
import org.eclipse.smila.jobmanager.DataObjectTypeDefinition;
import org.eclipse.smila.jobmanager.JobDefinition;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.JobManagerException;
import org.eclipse.smila.jobmanager.WorkerDefinition;
import org.eclipse.smila.jobmanager.WorkerDefinition.Input;
import org.eclipse.smila.jobmanager.WorkerDefinition.InputMode;
import org.eclipse.smila.jobmanager.WorkerDefinition.InputOutput;
import org.eclipse.smila.jobmanager.WorkerDefinition.Mode;
import org.eclipse.smila.jobmanager.WorkflowAction;
import org.eclipse.smila.jobmanager.WorkflowDefinition;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.jobmanager.persistence.PersistenceException;
import org.eclipse.smila.jobmanager.persistence.RunStorage;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGenerator;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorException;
import org.eclipse.smila.jobmanager.taskgenerator.TaskGeneratorProvider;
import org.eclipse.smila.jobmanager.util.ExpressionUtil;
import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.utils.collections.MultiValueMap;

/**
 * Class representing the data for a job run (the JobDefinition, the WorkflowDefinition, WorkerDefinitions, Buckets,
 * etc.).
 */
public class JobRun {

  /** if this time is exceeded while generating tasks, a log message is written. */
  private static final long MIN_TASK_GENERATION_TIME_TO_LOG = 1000; // ms

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** The id of the job run. */
  private final String _jobRunId;

  /** The JobDefinition of the job run. */
  private final JobDefinition _jobDef;

  /** The WorkflowDefinition of the job run's job. */
  private final WorkflowDefinition _workflowDef;

  /** The startAction of the workflow. */
  private WorkflowAction _startAction;

  /** bucket name -> workflow actions having this bucket as input in this JobRun. */
  private final MultiValueMap<String, WorkflowAction> _bucketToTriggeredAction =
    new MultiValueMap<String, WorkflowAction>();

  /** bucket name -> bucket instance (for this JobRun). */
  private final Map<String, Bucket> _buckets = new HashMap<String, Bucket>();

  /** WorkflowAction -> input slot name -> bucket. */
  private final Map<WorkflowAction, Map<String, Bucket>> _actionInputBuckets =
    new HashMap<WorkflowAction, Map<String, Bucket>>();

  /** WorkflowAction -> output slot name -> bucket. */
  private final Map<WorkflowAction, Map<String, Bucket>> _actionOutputBuckets =
    new HashMap<WorkflowAction, Map<String, Bucket>>();

  /** A map of the workers (key: worker-name, value: WorkerDefinition) for this job run. */
  private final Map<String, WorkerDefinition> _workers = new HashMap<String, WorkerDefinition>();

  /** A map of the merged parameters (job, workflow) (key: param-name, value: ValueExpression) for this job run. */
  private final AnyMap _parameters = DataFactory.DEFAULT.createAnyMap();

  /** A map with bucket names as key and bucket definition as value. */
  private final Map<String, BucketDefinition> _bucketDefinitions = new HashMap<String, BucketDefinition>();

  /** The provider from which to select a TaskGenerator. */
  private final TaskGeneratorProvider _taskGeneratorProvider;

  /**
   * Constructs the job run data for a job run.
   * 
   * @param runId
   *          the id of the run.
   * @param jobName
   *          The job name.
   * @param runStorage
   *          The runStorage.
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @param taskGeneratorProvider
   *          The TaskGeneratorProvider is used to select a TaskGenerator for generating new tasks
   * @throws Exception
   *           An exception if something goes wrong
   */
  public JobRun(final String runId, final String jobName, final RunStorage runStorage,
    final DefinitionPersistence definitions, final TaskGeneratorProvider taskGeneratorProvider) throws Exception {
    super();
    final AnyMap jobRunData = runStorage.getJobRunData(jobName, true);
    final AnyMap jobDefinitionMap = jobRunData.getMap(JobManagerConstants.DATA_JOB_RUN_JOB_DEF);
    final AnyMap workflowDefinitionMap = jobRunData.getMap(JobManagerConstants.DATA_JOB_RUN_WORKFLOW_DEF);
    final AnySeq bucketsSeq = jobRunData.getSeq(JobManagerConstants.DATA_JOB_RUN_BUCKET_DEFS);
    if (bucketsSeq != null && bucketsSeq.size() > 0) {
      final Iterator<Any> bucketsIterator = bucketsSeq.iterator();
      while (bucketsIterator.hasNext()) {
        final AnyMap bucketsAny = (AnyMap) bucketsIterator.next();
        _bucketDefinitions.put(bucketsAny.getStringValue(BucketDefinition.KEY_NAME), new BucketDefinition(
          bucketsAny));
      }
    }
    _taskGeneratorProvider = taskGeneratorProvider;
    _jobRunId = runId;
    _jobDef = new JobDefinition(jobDefinitionMap);
    _workflowDef = new WorkflowDefinition(workflowDefinitionMap);
    compileWorkflow(definitions);
  }

  /**
   * Compiles the workflow, i.e. merges the parameters, loads the WorkerDefinitions, creates Bucket informations with
   * the merged parameters.
   * 
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private void compileWorkflow(final DefinitionPersistence definitions) throws PersistenceException {
    if (_jobDef.getParameters() != null) {
      _parameters.putAll(_jobDef.getParameters());
    }
    if (_workflowDef.getParameters() != null) {
      _parameters.putAll(_workflowDef.getParameters());
    }
    _startAction = _workflowDef.getStartAction();
    compileWorkflowAction(_startAction, definitions);
    if (_workflowDef.getActions() != null) {
      for (final WorkflowAction action : _workflowDef.getActions()) {
        compileWorkflowAction(action, definitions);
      }
    }
  }

  /**
   * Compiles a single WorkflowAction (i.e. loads the WorkerDefinitions, creates Bucket informations with the merged
   * parameters).
   * 
   * If the isStartAction flag is set to true and the actions worker has 'bulkSource' mode, the worker is marked as
   * being able to start initial tasks.
   * 
   * @param action
   *          The WorkflowAction to compile.
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private void compileWorkflowAction(final WorkflowAction action, final DefinitionPersistence definitions)
    throws PersistenceException {
    final String workerName = action.getWorker();
    final WorkerDefinition workerDef = getWorkerDefinition(definitions, workerName);
    _actionInputBuckets.put(action, new HashMap<String, Bucket>());
    _actionOutputBuckets.put(action, new HashMap<String, Bucket>());

    // input
    final Map<String, String> actionInputSlotsDefinition = action.getInput();
    final Collection<? extends InputOutput<?>> workerInputSlots = workerDef.getInput();
    compileActionSlotConfigurations(action, definitions, actionInputSlotsDefinition, workerInputSlots, true);
    if (actionInputSlotsDefinition != null) {
      // store input bucket name to action
      for (final String bucketName : actionInputSlotsDefinition.values()) {
        _bucketToTriggeredAction.add(bucketName, action);
      }
    }

    // output
    final Map<String, String> actionOutputSlotsDefinition = action.getOutput();
    final Collection<? extends InputOutput<?>> workerOutputSlots = workerDef.getOutput();
    compileActionSlotConfigurations(action, definitions, actionOutputSlotsDefinition, workerOutputSlots, false);
  }

  /**
   * Compiles the slots (Input or Output) of a single WorkflowAction (i.e. loads the WorkerDefinitions, creates Bucket
   * informations with the merged parameters).
   * 
   * @param action
   *          The WorkflowAction of the slots.
   * @param definitions
   *          The definition persistence where the Bucket and Data Object Type definitions are stored.
   * @param actionSlots
   *          A map of the actions slot to bucket definition
   * @param workerSlots
   *          The slots of a worker
   * @param isInputSlots
   *          true if we are handling input slots
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private void compileActionSlotConfigurations(final WorkflowAction action,
    final DefinitionPersistence definitions, final Map<String, String> actionSlots,
    final Collection<? extends InputOutput<?>> workerSlots, final boolean isInputSlots) throws PersistenceException {
    if (actionSlots != null) {
      for (final Map.Entry<String, String> slot : actionSlots.entrySet()) {
        final String slotName = slot.getKey();
        final String bucketName = slot.getValue();
        Bucket bucket = _buckets.get(bucketName);
        if (bucket == null) {
          BucketDefinition bucketDef = _bucketDefinitions.get(bucketName);
          boolean isPersistent = true;
          if (bucketDef == null) {
            final String slotDataObjectType = getSlotType(slotName, workerSlots);
            isPersistent = false;
            bucketDef = new BucketDefinition(bucketName, slotDataObjectType);
          }
          final DataObjectTypeDefinition dot = definitions.getDataObjectType(bucketDef.getDataObjectType());
          // Use compiled parameters. Job-Parameters and Workflow-Parameters, but not WorkflowActions, since
          // workflow-action parameters must not influence buckets
          bucket = new Bucket(bucketDef, dot, isPersistent, _parameters);
          _buckets.put(bucketName, bucket);
        }
        if (isInputSlots) {
          _actionInputBuckets.get(action).put(slotName, bucket);
        } else {
          _actionOutputBuckets.get(action).put(slotName, bucket);

        }
      }
    }
  }

  /**
   * Returns the data object type of a slot with the given name.
   * 
   * @param slotName
   *          the name of the slot
   * @param slots
   *          a collection of Input or Output elements (describing the slots of the workers)
   * @return The data object type of a slot with the given name.
   */
  private String getSlotType(final String slotName, final Collection<? extends InputOutput<?>> slots) {
    if (slots != null) {
      for (final InputOutput<?> iterableElement : slots) {
        if (iterableElement.getName().equals(slotName)) {
          return iterableElement.getType();
        }
      }
    }
    return null;
  }

  /**
   * Gets a WorkerDefinition from a given DefinitionPersistence instance.
   * 
   * @param definitions
   *          The DefinitionPersistence where the definitions can be retrieved (e.g. WorkerDefinition).
   * @param workerName
   *          The name of the WorkerDefinition to load.
   * @return the WorkerDefinition for workerName.
   * @throws PersistenceException
   *           exception while accessing definitions in the DefinitionPersistence.
   */
  private WorkerDefinition getWorkerDefinition(final DefinitionPersistence definitions, final String workerName)
    throws PersistenceException {
    WorkerDefinition workerDef = _workers.get(workerName);
    if (workerDef == null) {
      workerDef = definitions.getWorker(workerName);
      _workers.put(workerName, workerDef);
    }
    return workerDef;
  }

  /**
   * @param workerName
   *          The name of the WorkerDefinition.
   * @return The WorkerDefinition with the name workerName.
   */
  public WorkerDefinition getWorkerDefinition(final String workerName) {
    return _workers.get(workerName);
  }

  /**
   * @return name of job definition
   */
  public String getJobName() {
    return _jobDef.getName();
  }

  /**
   * @return The ID of the job run.
   */
  public String getJobRunId() {
    return _jobRunId;
  }

  /**
   * @return The JobDefinition for the JobRun.
   */
  public JobDefinition getJobDefinition() {
    return _jobDef;
  }

  /**
   * @return The WorkflowDefinition for the JobRun.
   */
  public WorkflowDefinition getWorkflowDefinition() {
    return _workflowDef;
  }

  /**
   * @return The Collection of Bucket instances of the JobRun.
   */
  public Collection<Bucket> getBuckets() {
    return Collections.unmodifiableCollection(_buckets.values());
  }

  /**
   * 
   * @param bucketName
   *          the bucket's name.
   * @return the job's Bucket instance for a bucket with the given name
   */
  public Bucket getBucket(final String bucketName) {
    return _buckets.get(bucketName);
  }

  /**
   * @param action
   *          the action for which the input bucket should be looked up.
   * @param slotName
   *          an input slot name of this worker
   * @return the associated bucket, or null, if worker is not used in workflow, or the slot name is not an input slot.
   */
  public Bucket getBucketForInputSlot(final WorkflowAction action, final String slotName) {
    return _actionInputBuckets.get(action).get(slotName);
  }

  /**
   * @param action
   *          the action for which the input bucket should be looked up.
   * @param slotName
   *          an output slot name of this worker
   * @return the associated bucket, or null, if worker is not used in workflow, or the slot name is not an output slot.
   */
  public Bucket getBucketForOutputSlot(final WorkflowAction action, final String slotName) {
    return _actionOutputBuckets.get(action).get(slotName);
  }

  /**
   * Creates the initial task for this job run's new workflow run.
   * 
   * @param workerName
   *          The name of the worker requesting the initial task.
   * @param workflowRunId
   *          The id for the new workflow run.
   * @return An initial task for the new job run.
   * @throws JobManagerException
   *           An error occurred while checking if an initial task could be created or while creating the initial task.
   */
  public Task getInitialTask(final String workerName, final String workflowRunId) throws JobManagerException {
    if (_startAction == null || !workerName.equals(_startAction.getWorker())
      || !_workers.get(workerName).getModes().contains(Mode.BULKSOURCE)) {
      throw new JobManagerException("Worker '" + workerName + "' is not defined as start action for job '"
        + _jobDef.getName() + "' or has no '" + Mode.BULKSOURCE.toString() + "' mode.");
    }
    final List<Task> taskList = generateTasksForAction(workflowRunId, null, _startAction, null);
    // there must be only one task
    if (taskList.size() != 1) {
      throw new JobManagerException("Not exactly one initial task produced for worker '" + workerName
        + "' in workflow run '" + workflowRunId + "'.");
    }
    return taskList.get(0);
  }

  /**
   * Gets the follow-up tasks for workers within the actual workflow run depending on a list of bulks.
   * 
   * @param workflowRunId
   *          The id of the current workflow run.
   * @param bucketNameToReallyCreatedBulksMap
   *          The bulks potentially triggering tasks for follow up workers.
   * @param taskParamsToCopy
   *          the parameters from the current task to copy to the follow up tasks
   * @return A collection of tasks triggered by the bulks within this workflow for the given workflow run.
   * @throws TaskGeneratorException
   *           error while generating new tasks
   */
  public List<Task> getFollowupTasks(final String workflowRunId,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap, final AnyMap taskParamsToCopy)
    throws TaskGeneratorException {
    final List<Task> followUpTasks = new ArrayList<Task>();
    // check for follow up workflow actions for given bulks
    final Collection<WorkflowAction> followUpActions =
      getFollowupWorkflowActions(bucketNameToReallyCreatedBulksMap.keySet());

    // for all follow up actions: find appropriate TaskGenerator and generate follow up tasks
    for (final WorkflowAction action : followUpActions) {
      followUpTasks.addAll(generateTasksForAction(workflowRunId, bucketNameToReallyCreatedBulksMap, action,
        taskParamsToCopy));
    }
    return followUpTasks;
  }

  /**
   * get initial tasks for run once job run.
   */
  public List<Task> getInitialRunOnceTasks(final String workflowRunId) throws TaskGeneratorException {
    final TaskGenerator taskGenerator = getTaskGenerator(_startAction);

    // get start action parameters
    final AnyMap paramExpressions = DataFactory.DEFAULT.createAnyMap();
    if (_parameters != null) {
      paramExpressions.putAll(_parameters);
    }
    if (_startAction.getParameters() != null) {
      paramExpressions.putAll(_startAction.getParameters());
    }
    final AnyMap parameters = ExpressionUtil.evaluateParameters(paramExpressions);
    final Map<String, Bucket> inputSlotNameToBucket = _actionInputBuckets.get(_startAction);
    final Map<String, Bucket> outputSlotNameToBucketMap = _actionOutputBuckets.get(_startAction);

    final List<Task> tasks =
      taskGenerator.createRunOnceTasks(inputSlotNameToBucket, outputSlotNameToBucketMap, parameters,
        _startAction.getWorker());
    setAdditionalTaskProperties(tasks, workflowRunId, _startAction.getWorker());
    return tasks;
  }

  /**
   * @param triggeringBucket
   *          a bucket changed by another workflow
   * @return true if this job run can be triggered by this bucket.
   */
  public boolean isTriggeredBy(final Bucket triggeringBucket) {
    if (_startAction.getInput() != null) {
      for (final String inputBucketName : _startAction.getInput().values()) {
        final Bucket inputBucket = getBucket(inputBucketName);
        if (inputBucket.isPersistent() && triggeringBucket.getBucketId().equals(inputBucket.getBucketId())) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * @return persistent input buckets of start action.
   */
  public Collection<Bucket> getTriggerBuckets() {
    final Collection<Bucket> startBuckets = new ArrayList<Bucket>();
    if (_startAction.getInput() != null) {
      for (final String inputBucketName : _startAction.getInput().values()) {
        final Bucket inputBucket = getBucket(inputBucketName);
        if (inputBucket.isPersistent()) {
          startBuckets.add(inputBucket);
        }
      }
    }
    return startBuckets;
  }

  /**
   * Gets the follow-up tasks for workers within the actual workflow run depending on a list of bulks.
   * 
   * Note that all tasks returned by this method will have thir workflowRunId parameter set to "null" since there will
   * be no workflow run started. You will have to set this parameter for any task in the returned collection.
   * 
   * @param bucketNameToReallyCreatedBulksMap
   *          The bulks potentially triggering tasks for follow up workers.
   * @return A collection of tasks triggered by the bulks within this workflow for the given workflow run.
   * @throws TaskGeneratorException
   *           error while generating new tasks
   */
  public List<Task> getTriggeredInitialTasks(final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap)
    throws TaskGeneratorException {
    // only for the initial action
    // note: we are setting null as workflow run, the workflow-run-id has to be set outside!
    return generateTasksForAction(null, bucketNameToReallyCreatedBulksMap, _startAction, null);
  }

  /**
   * Generates a task for an action.
   * 
   * @param workflowRunId
   *          the workflow run id
   * @param bucketNameToReallyCreatedBulksMap
   *          the map from bucket name as key to BulkInfo-List as value
   * @param action
   *          the action to generate the tasks for
   * @param taskParamsToCopy
   *          the parameters from the current task to copy to the follow up tasks
   * @return A list of tasks for an action
   * @throws TaskGeneratorException
   *           error while generating new tasks
   */
  private List<Task> generateTasksForAction(final String workflowRunId,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap, final WorkflowAction action,
    final AnyMap taskParamsToCopy) throws TaskGeneratorException {

    final TaskGenerator taskGenerator = getTaskGenerator(action);

    // get parameters
    final AnyMap paramExpressions = DataFactory.DEFAULT.createAnyMap();
    if (_parameters != null) {
      paramExpressions.putAll(_parameters);
    }
    if (action.getParameters() != null) {
      paramExpressions.putAll(action.getParameters());
    }
    final AnyMap parameters = ExpressionUtil.evaluateParameters(paramExpressions);
    if (taskParamsToCopy != null) {
      parameters.putAll(taskParamsToCopy);
    }

    final Map<String, Bucket> inputSlotNameToBucket = _actionInputBuckets.get(action);
    final Map<String, Bucket> outputSlotNameToBucketMap = _actionOutputBuckets.get(action);

    final MultiValueMap<String, BulkInfo> changedInputSlotToBulkListMap =
      prepareInputSlots(action, bucketNameToReallyCreatedBulksMap);

    // let the task generator create the tasks
    final long start = System.currentTimeMillis();
    final List<Task> tasks =
      taskGenerator.createTasks(changedInputSlotToBulkListMap, inputSlotNameToBucket, outputSlotNameToBucketMap,
        parameters, action.getWorker());
    final long t = System.currentTimeMillis() - start;
    if (t > MIN_TASK_GENERATION_TIME_TO_LOG) {
      _log.info("TaskGenerator time for creating tasks: " + t);
    }

    setAdditionalTaskProperties(tasks, workflowRunId, action.getWorker());

    if (taskParamsToCopy != null) {
      for (final Task followUpTask : tasks) {
        for (final Entry<String, Any> param : taskParamsToCopy.entrySet()) {
          followUpTask.getParameters().put(param.getKey(), param.getValue());
        }
      }
    }

    return tasks;
  }

  /**
   * set task properties needed by JobManager for task-to-workflow association, and set qualifier, if the worker
   * requires one.
   * 
   * @param tasks
   *          new tasks
   * @param workflowRunId
   *          id of workflow run in which these task was created.
   * @param workerName
   *          name of worker
   */
  private void setAdditionalTaskProperties(final List<Task> tasks, final String workflowRunId,
    final String workerName) {
    final WorkerDefinition worker = getWorkerDefinition(workerName);
    String qualifierSlot = null;
    for (final Input input : worker.getInput()) {
      if (input.getModes().contains(InputMode.QUALIFIER)) {
        qualifierSlot = input.getName();
      }
    }
    for (final Task task : tasks) {
      task.getProperties().put(Task.PROPERTY_WORKFLOW_RUN_ID, workflowRunId);
      task.getProperties().put(Task.PROPERTY_JOB_RUN_ID, _jobRunId);
      task.getProperties().put(Task.PROPERTY_JOB_NAME, _jobDef.getName());
      if (qualifierSlot != null) {
        final List<BulkInfo> inputBulks = task.getInputBulks().get(qualifierSlot);
        if (inputBulks != null && !inputBulks.isEmpty()) {
          final BulkInfo firstBulk = inputBulks.get(0);
          task.setQualifier(firstBulk.getObjectName());
        }
      }
    }
  }

  /**
   * @param action
   *          the action to generate the tasks for
   * @return task generator configured for this action, or default generator.
   */
  private TaskGenerator getTaskGenerator(final WorkflowAction action) {
    final TaskGenerator taskGenerator;
    final WorkerDefinition worker = _workers.get(action.getWorker());
    if (worker.getTaskGenerator() != null && !worker.getTaskGenerator().equals("")) {
      taskGenerator = _taskGeneratorProvider.getTaskGenerator(worker.getTaskGenerator());
    } else {
      taskGenerator = _taskGeneratorProvider.getDefaultTaskGenerator();
    }
    return taskGenerator;
  }

  /**
   * associate the new input bulks that triggered this action with the slotnames.
   * 
   * @param action
   *          the action to generate the tasks for
   * @param bucketNameToReallyCreatedBulksMap
   *          the map from bucket name as key to BulkInfo-List as value
   * @return changedInputSlotToBulkListMap
   */
  private MultiValueMap<String, BulkInfo> prepareInputSlots(final WorkflowAction action,
    final MultiValueMap<String, BulkInfo> bucketNameToReallyCreatedBulksMap) {
    // prepare changed input buckets for new task
    final MultiValueMap<String, BulkInfo> changedInputSlotToBulkListMap = new MultiValueMap<String, BulkInfo>();
    final Map<String, String> actionInput = action.getInput();
    if (actionInput != null) {
      for (final Map.Entry<String, String> actionInputEntry : actionInput.entrySet()) {
        final List<BulkInfo> bulkInfoList = bucketNameToReallyCreatedBulksMap.get(actionInputEntry.getValue());
        if (bulkInfoList != null) {
          for (final BulkInfo inputBulk : bulkInfoList) {
            if (inputBulk != null) {
              if (!changedInputSlotToBulkListMap.containsKey(actionInputEntry.getKey())) {
                changedInputSlotToBulkListMap.put(actionInputEntry.getKey(), new ArrayList<BulkInfo>());
              }
              changedInputSlotToBulkListMap.get(actionInputEntry.getKey()).add(inputBulk);
            }
          }
        }
      }
    }
    return changedInputSlotToBulkListMap;
  }

  /**
   * Returns a set of follow up actions within a workflow. Follow up actions have to be unique!
   * 
   * @param bucketNames
   *          a collection of bucket names,.
   * @return A set of unique follow-up actions. Empty (but not null) if there are none.
   */
  private Collection<WorkflowAction> getFollowupWorkflowActions(final Collection<String> bucketNames) {
    final Set<WorkflowAction> followUpActions = new HashSet<WorkflowAction>();

    for (final String bucketName : bucketNames) {
      final List<WorkflowAction> actions = _bucketToTriggeredAction.get(bucketName);
      if (actions != null) {
        followUpActions.addAll(actions);
      }
    }
    return followUpActions;
  }

}
