/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.worker;

import java.util.List;

import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.taskmanager.TaskmanagerException;
import org.eclipse.smila.taskworker.RecoverableTaskException;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;

/**
 * Worker for asynchronous finishing of tasks.
 */
public class FinishTaskWorker implements Worker {

  /** worker name. */
  private static final String WORKER_NAME = TaskManager.FINISHING_TASKS_WORKER;

  /** the job manager to use for checking job state. */
  private JobManager _jobManager;

  /** the task manager to use for adding tasks. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return WORKER_NAME;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    try {
      final List<Task> newTasks = finishTask(taskContext);
      countNewTasks(taskContext, newTasks);
    } catch (final JobManagerException ex) {
      if (ex.isRecoverable()) {
        throw new RecoverableTaskException(ex.getMessage(), ex);
      } else {
        throw ex;
      }
    }
  }

  /** finish task and measure duration. */
  private List<Task> finishTask(final TaskContext taskContext) throws JobManagerException {
    taskContext.addToCounter("finishedTaskCount", 1);
    final long start = taskContext.getTimestamp();
    try {
      return _jobManager.finishTask(taskContext.getTask());
    } finally {
      taskContext.measureTime("finishTask", start);
    }
  }

  /** set counter for new tasks. */
  private void countNewTasks(final TaskContext taskContext, final List<Task> newTasks) throws TaskmanagerException {
    if (newTasks != null && !newTasks.isEmpty()) {
      taskContext.addToCounter("newTaskCount", newTasks.size());
    }
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }

  /**
   * set task manager to use before activation.
   * 
   * @param taskManager
   *          the task manager
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * remove task manager after deactivation.
   * 
   * @param taskManager
   *          the task manager
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (taskManager == _taskManager) {
      _taskManager = null;
    }
  }
}
