/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher (Attensity Europe GmbH) - implementation
 **********************************************************************************************************************/
package org.eclipse.smila.processing.bpel;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.eclipse.smila.datamodel.xml.XmlSerializationUtils;

/** scan BPEL definitions for invocations of other pipelines and find process name. */
public class BpelScanner {

  /** XML parser. */
  private XMLStreamReader _reader;

  /** true, if reader is at a invoke start element. */
  private boolean _isAtInvokeElement;

  /** name of BPEL process. */
  private String _processName;

  /** create instance for given BPEL string. */
  public BpelScanner(final String bpelDefinition) throws UnsupportedEncodingException, XMLStreamException {
    this(bpelDefinition.getBytes("UTF-8"));
  }

  /** create instance for given BPEL definition. */
  public BpelScanner(final byte[] bpelDefinition) throws XMLStreamException {
    this(new ByteArrayInputStream(bpelDefinition));
  }

  /** create instance for given BPEL stream. */
  public BpelScanner(final InputStream bpelDefinition) throws XMLStreamException {
    _reader = XmlSerializationUtils.getStaxReaderFactory().createXMLStreamReader(bpelDefinition);
  }

  /** value of name attribute of process element. Set after parsing BPEL has been done. */
  public String getProcessName() {
    return _processName;
  }

  /**
   * find next <code>&lt;invoke&gt;</code> element.
   * 
   * @return true if another one was found, otherwise false.
   */
  public boolean findNextInvoke() throws XMLStreamException {
    _isAtInvokeElement = false;
    if (_reader != null) {
      while (!_isAtInvokeElement && _reader.hasNext()) {
        final int event = _reader.next();
        if (event == XMLStreamReader.START_ELEMENT) {
          final String tagName = _reader.getLocalName();
          if (BPELConstants.TAG_INVOKE.equals(tagName)) {
            _isAtInvokeElement = true;
          } else if (BPELConstants.TAG_PROCESS.equals(tagName)) {
            _processName = _reader.getAttributeValue(null, BPELConstants.ATTR_NAME);
          }
        }
      }
      if (!_reader.hasNext()) {
        close();
      }
    }
    return _isAtInvokeElement;
  }

  /**
   * @return value of <code>portType</code> attribute if an <code>&lt;invoke&gt;</code> element was found. null if
   *         called before {@link #findNextInvoke()} was called or if it has returned false.
   */
  public String getCurrentPortType() {
    if (_reader != null && _isAtInvokeElement) {
      return _reader.getAttributeValue(null, BPELConstants.ATTR_PORTTYPE);
    }
    return null;
  }

  /**
   * @return value of <code>operation</code> attribute if an <code>&lt;invoke&gt;</code> element was found. null if
   *         called before {@link #findNextInvoke()} was called or if it has returned false.
   */
  public String getCurrentOperation() {
    if (_reader != null && _isAtInvokeElement) {
      return _reader.getAttributeValue(null, BPELConstants.ATTR_OPERATION);
    }
    return null;
  }

  /**
   * @return value of <code>partnerLink</code> attribute if an <code>&lt;invoke&gt;</code> element was found. null if
   *         called before {@link #findNextInvoke()} was called or if it has returned false.
   */
  public String getCurrentPartnerLink() {
    if (_reader != null && _isAtInvokeElement) {
      return _reader.getAttributeValue(null, BPELConstants.ATTR_PARTNERLINK);
    }
    return null;
  }

  /** close XML parser. */
  public void close() {
    if (_reader != null) {
      try {
        _reader.close();
      } catch (final Exception ex) {
        ; // ignore.
      } finally {
        _reader = null;
        _isAtInvokeElement = false;
      }
    }
  }
}
