package org.eclipse.smila.processing.bpel;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.WorkflowProcessor;
import org.eclipse.smila.zookeeper.ZkConcurrentMap;
import org.eclipse.smila.zookeeper.ZkConnection;
import org.eclipse.smila.zookeeper.ZooKeeperService;

/**
 * Uses ZooKeeper to coordinate workflow updating in a SMILA cluster.
 */
public class WorkflowUpdateWatcher {
  /** znode name to use for our data. */
  private static final String ROOTPATH = "/smila/processor/workflows";

  /** interval to poll for changes, in seconds. */
  private static final int POLL_INTERVAL = 60;

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** ZooKeeper service reference. */
  private final ZooKeeperService _zkService;

  /** processor to notify about update from other nodes. */
  private final WorkflowProcessor _processor;

  /** conection to zookeeper. */
  private final ZkConnection _zk;

  /** executes {@link #checkWorkflowVersions()} regularly. */
  private ScheduledExecutorService _scheduler;

  /** watcher implementation that calls {@link #checkWorkflowVersions()} and reinstalls the watch. */
  private class ZkWatcher implements Watcher {
    @Override
    public void process(final WatchedEvent event) {
      _watcherInstalled = false;
      checkWorkflowVersions();
      installWatch();
    }
  }

  /** ZkWatcher instance. */
  private final Watcher _watcher = new ZkWatcher();

  /** true if a watch is currently installed. */
  private boolean _watcherStarted;

  /** true if a watch is currently installed. */
  private boolean _watcherInstalled;

  /**
   * distributed map in ZooKeeper: each custom workflow is a key in this map, the ZK version of the key's node is used
   * to track updates.
   */
  private final ZkConcurrentMap _clusterVersions;

  /**
   * Versions of the znodes representing each workflow. If the actual version of the znode differs from this value, the
   * workflow should be updated on this node.
   */
  private final Map<String, Integer> _localVersions = new HashMap<String, Integer>();

  /**
   * create instance.
   * 
   * @throws ProcessingException
   *           error initializing the root node.
   */
  public WorkflowUpdateWatcher(final ZooKeeperService zkService, final WorkflowProcessor processor)
    throws ProcessingException {
    super();
    _zkService = zkService;
    _processor = processor;
    _zk = new ZkConnection(_zkService);
    try {
      _zk.ensurePathExists(ROOTPATH);
      _clusterVersions = new ZkConcurrentMap(_zk, ROOTPATH);
    } catch (final KeeperException ex) {
      throw new ProcessingException("Failed to create znode " + ROOTPATH, ex);
    }

  }

  /**
   * install a ZK watch on the root node to get notifications about changes.
   * 
   * @return true if watched was installed successfully
   */
  public synchronized boolean startWatching() {
    _watcherStarted = true;
    _watcherInstalled = false;
    installWatch();
    return _watcherInstalled;
  }

  /**
   * stop watching: ZK watch on root node will not be reinstalled. Does not remove a currently installed watch, so it's
   * possible that remaining watches will still receive notifications after this call.
   */
  public synchronized void stopWatching() {
    _watcherStarted = false;
    _watcherInstalled = false;
  }

  /** start polling for updates. */
  public void startPolling() {
    startPolling(POLL_INTERVAL);
  }

  /** start polling for updates with custom interval. */
  public synchronized void startPolling(final int pollIntervalSeconds) {
    if (_scheduler == null) {
      _scheduler = Executors.newScheduledThreadPool(1);
      _scheduler.scheduleWithFixedDelay(new Runnable() {
        @Override
        public void run() {
          checkWorkflowVersions();
          // ensure that watch is installed if watching was started.
          installWatch();
        }
      }, pollIntervalSeconds, pollIntervalSeconds, TimeUnit.SECONDS);
      _log.info("Started: polling for workflow updates each " + pollIntervalSeconds + " seconds.");
    }
  }

  /** stop polling for updates. */
  public synchronized void stopPolling() {
    if (_scheduler != null) {
      _scheduler.shutdownNow();
      _scheduler = null;
      _log.info("Stopped.");
    }
  }

  /**
   * Initialize notification structure for workflow, call on initial load on service start. Does not trigger
   * notifications for other nodes, just ensure that the node is created.
   */
  public synchronized void workflowLoadedOnStart(final String workflowName, final String timestamp)
    throws ProcessingException {
    try {
      _clusterVersions.putIfAbsent(workflowName, timestamp);
      _localVersions.put(workflowName, _clusterVersions.getVersion(workflowName));
    } catch (final RuntimeException ex) {
      throw new ProcessingException("Failed to initialize for workflow '" + workflowName + "'.");
    }
  }

  /**
   * Send update notification, call on custom workflow creation or update.
   */
  public synchronized void workflowUpdated(final String workflowName, final String timestamp)
    throws ProcessingException {
    try {
      final String oldTimestamp = _clusterVersions.getString(workflowName);
      Integer version;
      if (oldTimestamp == null) {
        _clusterVersions.put(workflowName, timestamp);
        version = _clusterVersions.getVersion(workflowName);
      } else {
        version = _clusterVersions.replaceAndGetVersion(workflowName, oldTimestamp, timestamp);
        if (version == null) {
          throw new ProcessingException("Failed to send update notification for workflow '" + workflowName + "'");
        }
      }
      _localVersions.put(workflowName, version);
      sendNotification(workflowName + " updated at " + timestamp);
    } catch (final RuntimeException ex) {
      throw new ProcessingException("Failed to update for workflow '" + workflowName + "'.");
    }
  }

  /**
   * Send delete notification, call on custom workflow remove.
   */
  public synchronized void workflowDeleted(final String workflowName) throws ProcessingException {
    try {
      _clusterVersions.remove(workflowName);
      _localVersions.remove(workflowName);
      sendNotification(workflowName + " deleted");
    } catch (final RuntimeException ex) {
      throw new ProcessingException("Failed to update for workflow '" + workflowName + "'.");
    }
  }

  /** compare cluster versions of workflows with local versions and update local deployment. */
  public synchronized void checkWorkflowVersions() {
    _log.debug("checking versions of workflows deployed in the cluster");
    try {
      final Set<String> obsoleteWorkflows = new HashSet<String>(_localVersions.keySet());
      for (final String workflowName : _clusterVersions.keySet()) {
        obsoleteWorkflows.remove(workflowName);
        checkWorkflowVersion(workflowName);
      }
      for (final String obsoleteWorkflow : obsoleteWorkflows) {
        deleteWorkflow(obsoleteWorkflow);
      }
    } catch (final Exception ex) {
      _log.warn(
        "Error getting cluster versions of workflows, maybe we are losing an update now. We'll retry later.", ex);
    }
  }

  /** compare cluster version of a single with local version and update local deployment. */
  private void checkWorkflowVersion(final String workflowName) {
    try {
      final Integer clusterVersion = _clusterVersions.getVersion(workflowName);
      if (clusterVersion != null) {
        final Integer localVersion = _localVersions.get(workflowName);
        if (localVersion == null || !clusterVersion.equals(localVersion)) {
          try {
            _processor.synchronizeWorkflowDefinition(workflowName, false);
            _localVersions.put(workflowName, clusterVersion);
          } catch (final Exception ex) {
            _log.warn("Error updating workflow '" + workflowName + "', old version will stay active.", ex);
          }
        }
      }
    } catch (final Exception ex) {
      _log.warn("Error getting cluster version of workflow '" + workflowName
        + "', maybe we are losing an update now.", ex);
    }
  }

  /** remove an obsolete workflow from the cluster. */
  private void deleteWorkflow(final String workflowName) {
    try {
      _processor.synchronizeWorkflowDefinition(workflowName, true);
      _localVersions.remove(workflowName);
    } catch (final Exception ex) {
      _log.warn("Error deleting workflow '" + workflowName + "', old version will stay active.", ex);
    }
  }

  /** touch root znode to trigger watches set by other nodes. */
  private void sendNotification(final String text) {
    try {
      _zk.setData(ROOTPATH, text.getBytes("utf-8"));
    } catch (final Exception ex) {
      _log.warn("Failed to update notification " + ROOTPATH + " for " + text
        + ". Watches may not be triggered, remove workflow updates may need more time.", ex);
    }
  }

  /** set watch on root node, if watching is enabled and no watch is supposed to be installed currently. */
  private synchronized void installWatch() {
    if (_watcherStarted && !_watcherInstalled) {
      try {
        _zk.exists(ROOTPATH, _watcher);
        _watcherInstalled = true;
      } catch (final KeeperException ex) {
        _log.warn("Starting to watch for updates failed, pipeline update notifications may take a bit longer: "
          + "Could not install watch on znode" + ROOTPATH, ex);
      }
    }
  }
}
