/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.io;

import java.util.HashMap;
import java.util.Map;

import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.objectstore.util.ObjectStoreRetryUtil;

import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.Counters;

/**
 * common base class of input/output wrappers. Apart from containing the defining {@link BulkInfo} object and providing
 * access to the bulk store, it does some basic IO duration measuring.
 */
public abstract class IODataObject {

  /** reference of objectstore service. */
  protected final ObjectStoreService _objectStore;

  /** the wrapped input data object. */
  private final BulkInfo _dataObject;

  /** time for opening the data object. */
  private long _durationOpen;

  /** time for performing IO on the data object. */
  private long _durationPerform;

  /** create instance. */
  public IODataObject(final BulkInfo dataObject, final ObjectStoreService objectStore) {
    _dataObject = dataObject;
    _objectStore = objectStore;
  }

  /** @return wrapped data object. */
  public BulkInfo getDataObject() {
    return _dataObject;
  }

  /** @return store name part of wrapped data object. */
  public String getStoreName() {
    return getDataObject().getStoreName();
  }

  /** @return object ID part of wrapped data object. */
  public String getObjectName() {
    return getDataObject().getObjectName();
  }

  /** @return reference to objectstore service. */
  public ObjectStoreService getObjectStore() {
    return _objectStore;
  }

  /** open store, retry on IOErrors. */
  protected void ensureStore() throws ObjectStoreException {
    ObjectStoreRetryUtil.retryEnsureStore(getObjectStore(), getStoreName());
  }

  /** create basic counter map. */
  public Map<String, Number> getCounter() {
    final Map<String, Number> counter = new HashMap<String, Number>();
    counter.put(Counters.DATAOBJECTCOUNT, 1);
    return counter;
  }

  /** @return duration needed to open the data object. */
  public long getDurationOpen() {
    return _durationOpen;
  }

  /** @return duration needed to perform IO operations on the data object. */
  public long getDurationPerform() {
    return _durationPerform;
  }

  /** @return current timestamp. */
  protected long startTime() {
    return System.nanoTime();
  }

  /** measure time for opening data object. */
  protected void timeOpen(final long startTime) {
    _durationOpen += (System.nanoTime() - startTime);
  }

  /** measure time for performing IO on data object. */
  protected void timePerform(final long startTime) {
    _durationPerform += (System.nanoTime() - startTime);
  }
}
