/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.workermanager.internal;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.Callable;

import org.eclipse.smila.objectstore.ServiceUnavailableException;
import org.eclipse.smila.taskmanager.ResultDescription;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskCompletionStatus;
import org.eclipse.smila.taskworker.Counters;
import org.eclipse.smila.taskworker.PostponeTaskException;
import org.eclipse.smila.taskworker.RecoverableTaskException;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.utils.MaybeRecoverableException;

/**
 * Encapsulates pair of worker/taskcontext for execution.
 */
public class WorkerRunner implements Callable<WorkerRunner> {

  /** the task context of the task to process. */
  private final TaskContext _taskContext;

  /** the worker to process the task. */
  private final Worker _worker;

  /** will be set when task processing is finished. */
  private ResultDescription _result;

  /**
   * @param worker
   *          the worker to process the task
   * @param taskContext
   *          the task context containing the task to process.
   */
  public WorkerRunner(final Worker worker, final TaskContext taskContext) {
    _worker = worker;
    _taskContext = taskContext;
  }

  @Override
  public WorkerRunner call() throws Exception {
    final long start = _taskContext.getTimestamp();
    try {
      _worker.perform(_taskContext);
      final long endPerform = _taskContext.getTimestamp();
      if (_taskContext.isCanceled()) {
        _result = new ResultDescription(TaskCompletionStatus.OBSOLETE, null, null, null);
        _taskContext.getOutputs().abortAll();
      } else {
        _taskContext.getOutputs().commitAll();
        final long endCommit = _taskContext.getTimestamp();
        final Map<String, Number> taskCounters = getCounters(start, endPerform, endCommit);
        _result = new ResultDescription(TaskCompletionStatus.SUCCESSFUL, null, null, taskCounters);
      }
    } catch (final Exception e) {
      _taskContext.getOutputs().abortAll();
      if (isRecoverableException(e)) {
        final String errorMessage =
          "Error while executing task " + _taskContext.getTask().getTaskId() + " in worker " + _worker + ": "
            + e.getMessage();
        _result = new ResultDescription(TaskCompletionStatus.RECOVERABLE_ERROR, "TaskWorker", errorMessage, null);
        _taskContext.getLog().warn(errorMessage, e);
      } else if (isPostponeException(e)) {
        final String errorMessage =
          "Error while executing task " + _taskContext.getTask().getTaskId() + " in worker " + _worker + ": "
            + e.getMessage();
        _result = new ResultDescription(TaskCompletionStatus.POSTPONE, "TaskWorker", errorMessage, null);
        _taskContext.getLog().warn(errorMessage, e);
      } else {
        final String errorMessage =
          "Error while executing task " + _taskContext.getTask().getTaskId() + " in worker " + _worker + ": "
            + e.getMessage();
        _result = new ResultDescription(TaskCompletionStatus.FATAL_ERROR, "TaskWorker", errorMessage, null);
        _taskContext.getLog().error(errorMessage, e);
      }
    } finally {
      _taskContext.getInputs().closeAll(); // closes all input quietly (-> will not throw an exception)
    }
    return this;
  }

  /** check if exception is a exception to create a recoverable error. */
  private boolean isRecoverableException(final Exception e) {
    if (e instanceof RecoverableTaskException || e instanceof IOException
      || e instanceof ServiceUnavailableException) {
      return true;
    }
    if (e instanceof MaybeRecoverableException) {
      return ((MaybeRecoverableException) e).isRecoverable();
    }
    return false;
  }

  /** check if exception is a exception to create a postponed task. */
  private boolean isPostponeException(final Exception e) {
    return e instanceof PostponeTaskException;
  }

  /**
   * 
   * @param startTime
   *          start of perform
   * @param endTime
   *          end of perform
   */
  private Map<String, Number> getCounters(final long startTime, final long endPerformTime, final long endTime) {
    final Map<String, Number> taskCounters = _taskContext.getFinalCounters();
    final long durationPerform = endPerformTime - startTime;
    final long durationTotal = endTime - startTime;
    Counters.addDuration(taskCounters, Counters.DURATION, durationTotal);
    Counters.addDuration(taskCounters, Counters.DURATION_PERFORM, durationPerform);
    // duration.iodata.open time is contained in our measured perform time, so we have to subtract it.
    final Number durationIoDataOpen = taskCounters.get(Counters.DURATION_IODATA_OPEN);
    if (durationIoDataOpen != null) {
      Counters.add(taskCounters, Counters.DURATION_PERFORM, -durationIoDataOpen.doubleValue());
    }
    return taskCounters;
  }

  /**
   * @return worker name
   */
  public String getWorkerName() {
    return _worker.getName();
  }

  /** @return task context. */
  public TaskContext getTaskContext() {
    return _taskContext;
  }

  /**
   * @return task
   */
  public Task getTask() {
    return _taskContext.getTask();
  }

  /**
   * @return result for executed task, "null" as long as execution isn't completed.
   */
  public ResultDescription getResult() {
    return _result;
  }
}
