package org.eclipse.smila.datamodel.ipc;

import java.io.IOException;
import java.io.InputStream;
import java.util.NoSuchElementException;

import org.apache.commons.io.IOUtils;
import org.eclipse.smila.datamodel.Any;

/**
 * Iterator-like class to read a sequence of BON {@link Any} objects from an input stream. Does not implement
 * {@link java.util.Iterator} because the methods throw {@link IOException} on read errors.
 * 
 * @author scum36
 * 
 */
public class BinaryObjectStreamIterator {

  /** the BON stream to read. */
  private final InputStream _binaryStream;

  /** BON parser. */
  private final IpcSerializationUtils _parser = new IpcSerializationUtils();

  /** next object to deliver. */
  private Any _nextObject;

  /** goes to true if end-of-stream reached. */
  private boolean _finished;

  /** create instance. */
  public BinaryObjectStreamIterator(final InputStream binaryStream) {
    _binaryStream = binaryStream;
  }

  /**
   * Check if more objects are available on the stream. Exception can only be thrown when called first after the
   * constructor or a call to {@link #next()}, consecutive calls to this method will not throw exception.
   * 
   * @return <code>true</code> if the next call to {@link #next()} will return an {@link Any} object, else
   *         <code>false</code>.
   * 
   * @throws IOException
   *           on stream read errors.
   * @throws IllegalStateException
   *           on BON parse errors.
   */
  public boolean hasNext() throws IOException {
    if (!_finished) {
      try {
        if (_nextObject == null) {
          _nextObject = _parser.binaryStream2any(_binaryStream);
        }
      } finally {
        if (_nextObject == null) {
          close();
        }
      }
    }
    return !_finished;
  }

  /**
   * Get next object from stream. Does not throw any exception if called next after a call to {@link #next()} that
   * returned true.
   * 
   * @return next object from stream
   * @throws NoSuchElementException
   *           no more objects available, a call to {@link #hasNext()} would have returned <code>false</code>.
   * @throws IOException
   *           on stream read errors.
   * @throws IllegalStateException
   *           on BON parse errors.
   */
  public Any next() throws IOException {
    if (_nextObject == null && !hasNext()) {
      throw new NoSuchElementException("No more objects available");
    }

    final Any result = _nextObject;
    _nextObject = null;
    return result;
  }

  /** close the underlying stream. */
  public void close() {
    _nextObject = null;
    _finished = true;
    IOUtils.closeQuietly(_binaryStream);
  }
}
