/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/

package org.eclipse.smila.bulkbuilder.helper;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.BulkbuilderService;
import org.eclipse.smila.bulkbuilder.WorkflowRunInfo;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.validation.InvalidRecordException;
import org.eclipse.smila.datamodel.validation.RecordValidator;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.events.JobEvent;
import org.eclipse.smila.jobmanager.events.JobListener;
import org.eclipse.smila.jobmanager.events.PrepareToFinishEvent;
import org.eclipse.smila.objectstore.ObjectStoreService;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;
import org.eclipse.smila.workermanager.TaskKeepAliveListener;
import org.eclipse.smila.workermanager.WorkerManager;
import org.osgi.service.component.ComponentContext;

/**
 * Base class for implementations of BulkBuilderService.
 */
public abstract class BulkbuilderServiceBase implements BulkbuilderService, JobListener, TaskKeepAliveListener {

  /** local Logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** flag if service is activated. */
  private boolean _isActive;

  /** the task manager to use for creating tasks. */
  private TaskManager _taskManager;

  /** the job manager to use for checking job state. */
  private JobManager _jobManager;

  /** the workermanager is used for task keep alive. */
  private WorkerManager _workerManager;

  /** connection to objectStore service. */
  private ObjectStoreService _objectStore;

  /** reference to the blackboard factory service. */
  private BlackboardFactory _blackboardFactory;

  /** bulk builder. */
  private BulkbuilderBase _bulkbuilder;

  /** initial task administration. */
  private BulkbuilderTaskProvider _taskProvider;

  /** to validate the incoming records. */
  private RecordValidator _recordValidator;

  /** bulk tracker watching time constraints for the streaming bulk builder. */
  private BulkTracker _bulkTracker;

  /** executor for scheduled monitor threads. */
  private final ScheduledExecutorService _scheduledMonitorExecutor = Executors.newScheduledThreadPool(3);

  /**
   * activate service with given bulkbuilder and validator. To be called from {@link #activate(ComponentContext)}
   * implementation in subclass.
   */
  protected void activateService(final BulkbuilderBase bulkbuilder, final BulkbuilderTaskProvider taskProvider,
    final RecordValidator validator) {
    _bulkbuilder = bulkbuilder;
    _taskProvider = taskProvider;
    _recordValidator = validator;
    _bulkTracker = new BulkTracker(_bulkbuilder);
    _scheduledMonitorExecutor.scheduleAtFixedRate(_bulkTracker, BulkTracker.TIME_LIMIT, BulkTracker.TIME_LIMIT,
      TimeUnit.MILLISECONDS);
    _workerManager.addKeepAliveListener(this);
    _isActive = true;
  }

  /**
   * @param context
   *          OSGi service component context.
   */
  protected abstract void activate(final ComponentContext context);

  /**
   * OSGi Declarative Services service deactivation method.
   * 
   * @param context
   *          OSGi service component context.
   */
  protected void deactivate(final ComponentContext context) {
    _isActive = false;
    _workerManager.removeKeepAliveListener(this);
    if (_bulkbuilder != null) {
      _bulkbuilder.shutdown();
    }
    if (_scheduledMonitorExecutor != null) {
      _scheduledMonitorExecutor.shutdown();
    }
  }

  @Override
  public WorkflowRunInfo addRecord(final String jobName, final Record record) throws BulkbuilderException,
    InvalidRecordException {
    checkActive();
    if (record == null) {
      throw new IllegalArgumentException("Error during adding record: Record must not be null");
    } else {
      _recordValidator.validate(record);
      return _bulkbuilder.addRecord(jobName, record);
    }
  }

  @Override
  public WorkflowRunInfo deleteRecord(final String jobName, final Record record) throws BulkbuilderException,
    InvalidRecordException {
    checkActive();
    if (record == null) {
      throw new IllegalArgumentException("Error during deleting record: Record must not be null");
    } else {
      _recordValidator.validate(record);
      return _bulkbuilder.deleteRecord(jobName, record);
    }
  }

  @Override
  public WorkflowRunInfo commitJob(final String jobName) throws BulkbuilderException {
    checkActive();
    return _bulkbuilder.commitBulk(jobName);
  }

  @Override
  public WorkflowRunInfo addToMicroBulk(final String jobName, final Record record, final String microBulkId)
    throws BulkbuilderException, InvalidRecordException {
    checkActive();
    if (record == null) {
      throw new IllegalArgumentException("Error during adding record: Record must not be null");
    } else {
      _recordValidator.validate(record);
      return _bulkbuilder.addToMicroBulk(jobName, record, microBulkId);
    }
  }

  @Override
  public WorkflowRunInfo finishMicroBulk(final String jobName, final String microBulkId)
    throws BulkbuilderException {
    checkActive();
    return _bulkbuilder.finishMicroBulk(jobName, microBulkId);
  }

  @Override
  public void removeMicroBulk(final String microBulkId) {
    _bulkbuilder.removeMicroBulk(microBulkId);
  }

  @Override
  public void processJobEvent(final JobEvent evt) {
    // before the job gets finished we have to commit the bulks for it.
    if (evt instanceof PrepareToFinishEvent) {
      if (_log.isDebugEnabled()) {
        _log.debug("Finish job '" + evt.getJobName() + "', committing bulks.");
      }
      // must not throw exceptions
      try {
        commitJob(evt.getJobName());
      } catch (final BulkbuilderException e) {
        _log.error("Error while finishing job '" + evt.getJobName() + "'.", e);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removedTask(final Task task) {
    // we'll just remove the task and log a warning.
    // this will prevent the task from being finished.
    if (_log.isWarnEnabled()) {
      _log.warn("Task '" + task.getTaskId() + "' has been removed by the system.");
    }
    _taskProvider.removeTask(task);
  }

  /**
   * check if service is active and throw an exception otherwise.
   * 
   * @throws BulkbuilderException
   *           if service is not active.
   */
  protected void checkActive() throws BulkbuilderException {
    if (!_isActive) {
      throw new BulkbuilderException("BulkBuilder is currently not active, probably the system is shutting down.");
    }
  }

  /** access objectstore service reference, if already set. */
  public ObjectStoreService getObjectStoreService() {
    return _objectStore;
  }

  /**
   * method for DS to set a service reference.
   * 
   * @param objectStoreService
   *          ObjectStoreService reference.
   */
  public void setObjectStoreService(final ObjectStoreService objectStoreService) {
    _objectStore = objectStoreService;
  }

  /**
   * method for DS to unset a service reference.
   * 
   * @param objectStoreService
   *          ObjectStoreService reference.
   */
  public void unsetObjectStoreService(final ObjectStoreService objectStoreService) {
    if (_objectStore == objectStoreService) {
      _objectStore = null;
    }
  }

  /** access taskmanager service reference, if already set. */
  public TaskManager getTaskManager() {
    return _taskManager;
  }

  /**
   * set task manager to use before activation.
   * 
   * @param taskManager
   *          the task manager
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * remove task manager after deactivation.
   * 
   * @param taskManager
   *          the task manager
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (taskManager == _taskManager) {
      _taskManager = null;
    }
  }

  /** access jobmanager service reference, if already set. */
  public JobManager getJobManager() {
    return _jobManager;
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }

  /** access objectstore service reference, if already set. */
  public WorkerManager getWorkerManager() {
    return _workerManager;
  }

  /**
   * @param workerManager
   *          the reference to the WorkerManager service.
   */
  public void setWorkerManager(final WorkerManager workerManager) {
    _workerManager = workerManager;
  }

  /**
   * @param workerManager
   *          the reference to the current WorkerManager service.
   */
  public void unsetWorkerManager(final WorkerManager workerManager) {
    if (_workerManager == workerManager) {
      _workerManager = null;
    }
  }

  /** returns the {@link BlackboardFactory} service reference set by the DS. */
  protected BlackboardFactory getBlackboardFactory() {
    return _blackboardFactory;
  }

  /**
   * Sets a reference to a blackboard factory before activation. Called by OSGi Declarative Services.
   */
  protected void setBlackboardFactory(final BlackboardFactory factory) {
    _blackboardFactory = factory;
    // we share our factory with the bulk builder.
    if (_bulkbuilder != null) {
      _bulkbuilder.setBlackboardFactory(factory);
    }
  }

  /**
   * Resets a reference to a blackboard factory. Called by OSGi Declarative Services.
   */
  protected void unsetBlackboardFactory(final BlackboardFactory factory) {
    if (_blackboardFactory == factory) {
      _blackboardFactory = null;
      // we share our factory with the bulk builder.
      _bulkbuilder.unsetBlackboardFactory(factory);
    }
  }
}
