/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Ivan Churkin (brox IT Solutions GmbH) - initial creator Sebastian Voigt (brox IT Solutions GmbH)
 * Andreas Weber (Attensity Europe GmbH) - changes for job management
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.connectivity.ConnectivityException;
import org.eclipse.smila.management.DeclarativeServiceManagementAgent;

/**
 * The Class CrawlerControllerAgent.
 */
public abstract class CrawlerControllerAgentBase extends DeclarativeServiceManagementAgent<CrawlerController>
  implements CrawlerControllerAgent {

  /**
   * The Constant LOG.
   */
  private final Log _log = LogFactory.getLog(CrawlerControllerAgentBase.class);

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.management.LocatedManagementAgentBase#getCategory()
   */
  @Override
  protected String getCategory() {
    return null;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.management.LocatedManagementAgentBase#getName()
   */
  @Override
  protected String getName() {
    return "CrawlerController";
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerControllerAgent#startCrawlerTask()
   */
  @Override
  public String startCrawlerTask(final String dataSourceId, final String jobName) {
    try {
      final int importRunId = _service.startCrawl(dataSourceId, jobName);
      return "Crawler with the dataSourceId '" + dataSourceId + "' pushing to job '" + jobName
        + "' successfully started! (import run id: " + importRunId + ")";
    } catch (final ConnectivityException exception) {
      _log.error(exception);
      return getExceptionText(exception);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerControllerAgent#stopCrawlerTask(java.lang.String)
   */
  @Override
  public String stopCrawlerTask(final String dataSourceId) {
    try {
      _service.stopCrawl(dataSourceId);
      return "Crawler with the dataSourceId '" + dataSourceId + "' successfully stopped.";
    } catch (final ConnectivityException exception) {
      _log.error(exception);
      return getExceptionText(exception);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerControllerAgent#getAgentTasksState()
   */
  @Override
  public Map<String, String> getCrawlerTasksState() {
    final HashMap<String, String> states = new HashMap<String, String>();
    final Map<String, CrawlState> crawlStates = _service.getCrawlerTasksState();
    for (final Entry<String, CrawlState> stateEntry : crawlStates.entrySet()) {
      states.put(stateEntry.getKey(), stateEntry.getValue().getState().name());
    }
    return states;
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerControllerAgent#getStatus(java.lang.String)
   */
  public String getStatus(final String dataSourceId) {
    return _service.getCrawlerTasksState().get(dataSourceId).getState().name();
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerControllerAgent#getAvailableCrawlers()
   */
  @Override
  public String[] getAvailableCrawlers() {
    final Collection<String> availCrawlers = _service.getAvailableCrawlers();
    return availCrawlers.toArray(new String[availCrawlers.size()]);
  }

  /**
   * {@inheritDoc}
   * 
   * @see org.eclipse.smila.connectivity.framework.CrawlerControllerAgent#getAvailableCrawlerTasks()
   */
  @Override
  public String[] getAvailableCrawlerTasks() {
    final Collection<String> configs = _service.getAvailableConfigurations();
    return configs.toArray(new String[configs.size()]);
  }

  /**
   * Returns the text of the exception plus any additional text from the exception's cause.
   * 
   * @param t
   *          the Throwable
   * @return the exception text
   */
  private String getExceptionText(final Throwable t) {
    String text = t.getMessage();
    final Throwable cause = t.getCause();
    if (cause != null) {
      text += ": " + cause.toString();
    }
    return text;
  }
}
