/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.httphandler;

import java.util.List;

import org.eclipse.smila.connectivity.framework.AgentController;
import org.eclipse.smila.connectivity.framework.AgentState;
import org.eclipse.smila.connectivity.framework.performancecounters.AgentPerformanceCounterHelper;
import org.eclipse.smila.connectivity.framework.performancecounters.ConnectivityPerformanceAgent;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;

/**
 * Implements the handling of HTTP requests to request the agents list. <br>
 * 
 * URL pattern: <code>smila/agents/</code> <br>
 * 
 * Methods allowed: <code>GET</code>
 */
public class AgentHandler extends BaseConnectivityHandler {

  /** the key for the job name. */
  public static final String KEY_JOB_NAME = "jobName";

  /** the key for the import run id. */
  public static final String KEY_IMPORT_RUN_ID = "importRunId";

  /** the key for the data source id. */
  public static final String KEY_DATA_SOURCE_ID = "dataSourceId";

  /** Reference to the {@link AgentController}. */
  private AgentController _agentController;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      // return statistics for last/current import run
      final String dataSourceId = getDataSourceId(requestUri);
      final AgentPerformanceCounterHelper<? extends ConnectivityPerformanceAgent> counterHelper =
        _agentController.getPerformanceCounterHelper(dataSourceId);
      if (counterHelper == null) {
        throw new ImportNotFoundException("No statistics found for data source id '" + dataSourceId + "'");
      }
      final ConnectivityPerformanceAgent instanceAgent = counterHelper.getInstanceAgent(); // last crawl run entry
      final AnyMap result = createStatistic(instanceAgent, dataSourceId);
      return result;
    } else if ("POST".equals(method)) {
      final String jobName;
      if (inputRecord != null) {
        jobName = inputRecord.getMetadata().getStringValue(KEY_JOB_NAME);
      } else {
        throw new IllegalArgumentException("No attribute '" + KEY_JOB_NAME + "' provided.");
      }
      final List<String> uriParts = getDynamicUriParts(requestUri);
      final String dataSourceId = uriParts.get(0);
      final int runId = _agentController.startAgent(dataSourceId, jobName);
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      result.put(KEY_IMPORT_RUN_ID, runId);
      return result;
    }
    return null;
  }

  /** {@inheritDoc} */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are allowed.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * @return the data source id from the request URL.
   */
  private String getDataSourceId(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * @return statistics returned for GET request.
   */
  private AnyMap createStatistic(final ConnectivityPerformanceAgent crawlerCounters, final String dataSourceId) {
    AnyMap result = null;
    if (crawlerCounters != null) {
      result = crawlerCounters.toAny();
    } else {
      result = DataFactory.DEFAULT.createAnyMap();
    }
    result.put(KEY_DATA_SOURCE_ID, dataSourceId);

    final AgentState state = _agentController.getAgentTasksState().get(dataSourceId);
    String stateString;
    if (state != null) {
      stateString = state.getState().name();
    } else {
      stateString = "Undefined";
    }
    result.put(CrawlersHandler.KEY_STATE, stateString);

    return result;
  }

  /**
   * OSGi method to set the {@link AgentController}. Called by DS.
   * 
   * @param agentController
   */
  protected void setAgentController(final AgentController agentController) {
    _agentController = agentController;
  }

  /**
   * OSGi method to unset the {@link AgentController}. Called by DS.
   * 
   * @param agentController
   */
  protected void unsetAgentController(final AgentController agentController) {
    if (_agentController == agentController) {
      _agentController = null;
    }
  }

}
