package org.eclipse.smila.connectivity.framework.httphandler;

import java.util.List;

import org.eclipse.smila.connectivity.framework.CrawlState;
import org.eclipse.smila.connectivity.framework.CrawlerController;
import org.eclipse.smila.connectivity.framework.performancecounters.ConnectivityPerformanceAgent;
import org.eclipse.smila.connectivity.framework.performancecounters.CrawlerPerformanceCounterHelper;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;

/**
 * Implements the handling of HTTP requests to request a single crawler with data source. <br>
 * 
 * URL pattern: <code>smila/crawlers/-dataSourceId-</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class CrawlerHandler extends BaseConnectivityHandler {

  /** the key for the job name. */
  public static final String KEY_JOB_NAME = "jobName";

  /** the key for the import run id. */
  public static final String KEY_IMPORT_RUN_ID = "importRunId";

  /** the key for the data source id. */
  public static final String KEY_DATA_SOURCE_ID = "dataSourceId";

  /** reference to CrawlerController service. */
  private CrawlerController _crawlerController;

  /** {@inheritDoc} */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      // return statistics for last/current import run
      final String dataSourceId = getDataSourceId(requestUri);
      final CrawlerPerformanceCounterHelper<? extends ConnectivityPerformanceAgent> counterHelper =
        _crawlerController.getCrawlerCounterHelper(dataSourceId);
      if (counterHelper == null) {
        throw new ImportNotFoundException("No statistics found for data source id '" + dataSourceId + "'");
      }
      final ConnectivityPerformanceAgent instanceAgent = counterHelper.getInstanceAgent(); // last crawl run entry
      final AnyMap result = createStatistic(instanceAgent, dataSourceId);
      return result;
    } else if ("POST".equals(method)) {
      final String jobName;
      if (inputRecord != null) {
        jobName = inputRecord.getMetadata().getStringValue(KEY_JOB_NAME);
      } else {
        throw new IllegalArgumentException("No attribute '" + KEY_JOB_NAME + "' provided.");
      }
      final List<String> uriParts = getDynamicUriParts(requestUri);
      final String dataSourceId = uriParts.get(0);
      final int runId = _crawlerController.startCrawl(dataSourceId, jobName);
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      result.put(KEY_IMPORT_RUN_ID, runId);
      return result;
    }
    return null;
  }

  /**
   * @return statistics returned for GET request.
   */
  private AnyMap createStatistic(final ConnectivityPerformanceAgent crawlerCounters, final String dataSourceId) {
    AnyMap result = null;
    if (crawlerCounters != null) {
      result = crawlerCounters.toAny();
    } else {
      result = DataFactory.DEFAULT.createAnyMap();
    }
    result.put(KEY_DATA_SOURCE_ID, dataSourceId);

    final CrawlState state = _crawlerController.getCrawlerTasksState().get(dataSourceId);
    String stateString;
    if (state != null) {
      stateString = state.getState().name();
    } else {
      stateString = "Undefined";
    }
    result.put(CrawlersHandler.KEY_STATE, stateString);

    return result;
  }

  /**
   * @return the data source id from the request URL.
   */
  private String getDataSourceId(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /** {@inheritDoc} */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * OSGi method to set the {@link CrawlerController}. Called by DS.
   * 
   * @param crawlerController
   */
  public void setCrawlerController(final CrawlerController crawlerController) {
    _crawlerController = crawlerController;
  }

  /**
   * OSGi method to unset the {@link CrawlerController}. Called by DS.
   * 
   * @param crawlerController
   */
  public void unsetCrawlerController(final CrawlerController crawlerController) {
    if (crawlerController == _crawlerController) {
      _crawlerController = null;
    }
  }

}
