/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.jobmanager.DataObjectTypeDefinition.Mode;
import org.eclipse.smila.jobmanager.util.IdGenerator;
import org.eclipse.smila.jobmanager.util.ValueExpression;
import org.eclipse.smila.taskmanager.BulkInfo;

/**
 * Class supporting the handling of a BucketDefinition.
 */
public class Bucket {
  /** The BucketDefinition. */
  private final BucketDefinition _bucketDef;

  /** The BucketDefinition's DataObjectTypeDefinition. */
  private final DataObjectTypeDefinition _dot;

  /** generated to identify persistent buckets created from same definition with same parameters. */
  private String _bucketId;

  /** final store name after applying job and bucket parameters. */
  private String _storeName;

  /** The bucket's mode. */
  private final Mode _bucketMode;

  /** The bucket's parameters. */
  private final AnyMap _bucketParameters = DataFactory.DEFAULT.createAnyMap();

  /**
   * Constructs a new Bucket.
   * 
   * @param bucketDef
   *          The BucketDefinition.
   * @param dot
   *          The DataObjectTypeDefinition for the BucketDefinition.
   * @param isPersistent
   *          'true' if the bucket is persistent, 'false' if it is transient.
   * @param parameters
   *          The parameters used for constructing the objects in the store.
   */
  public Bucket(final BucketDefinition bucketDef, final DataObjectTypeDefinition dot, final boolean isPersistent,
    final AnyMap parameters) {
    super();
    _bucketDef = bucketDef;
    _dot = dot;
    if (isPersistent) {
      _bucketMode = Mode.PERSISTENT;
    } else {
      _bucketMode = Mode.TRANSIENT;
    }
    applyParameters(parameters);
  }

  /**
   * @return The BucketDefinition of this Bucket.
   */
  public BucketDefinition getBucketDefinition() {
    return _bucketDef;
  }

  /**
   * @return The DataObjectTypeDefinition of this Bucket.
   */
  public DataObjectTypeDefinition getDataObjectTypeDefinition() {
    return _dot;
  }

  /**
   * @return The (generated) id of the bucket.
   */
  public String getBucketId() {
    return _bucketId;
  }

  /**
   * @return The name of the store.
   */
  public String getStoreName() {
    return _storeName;
  }

  /**
   * @return 'true' if the bucket is persistent, 'false' if not.
   */
  public boolean isPersistent() {
    return _bucketMode == Mode.PERSISTENT;
  }

  /**
   * @return 'true' if the bucket is transient, 'false' if not.
   */
  public boolean isTransient() {
    return _bucketMode == Mode.TRANSIENT;
  }

  /**
   * @return the bucket mode (TRANSIENT / PERSISTENT)
   */
  public Mode getMode() {
    return _bucketMode;
  }

  /**
   * Applies the given external parameters to the object to create the ID, the store and bucket name.
   * 
   * @param jobParameters
   *          the external parameters to apply.
   */
  private void applyParameters(final AnyMap jobParameters) {
    mergeParameters(jobParameters);
    _bucketId = createBucketId(_bucketParameters);
    _storeName = _dot.getStore(_bucketMode).evaluate(_bucketParameters).getExpression();
    _bucketParameters.put("_bucketName", _bucketDef.getName());
    _bucketParameters.put("_store", _storeName);
  }

  /**
   * Merges external parameters given from the outside world (e.g. job parameters) with the defined parameters from the
   * BucketDefinition. Parameters defined in the BucketDefinition override external parameters.
   * 
   * @param jobParameters
   *          The external parameters to merge with the internal parameters.
   */
  private void mergeParameters(final AnyMap jobParameters) {
    if (jobParameters != null) {
      _bucketParameters.putAll(jobParameters);
    }
    if (_bucketDef != null) {
      final AnyMap bucketDefParams = _bucketDef.getParameters();
      if (bucketDefParams != null) {
        _bucketParameters.putAll(bucketDefParams);
      }
    }
  }

  /**
   * Creates and returns a buckets id from the given job parameters.
   * 
   * @param jobParameters
   *          the parameters as defined in the job.
   * @return the created bucket id.
   */
  private String createBucketId(final AnyMap jobParameters) {
    final Collection<String> variables = getBucketIdVariables();
    final StringBuilder idBuilder = new StringBuilder(_bucketDef.getName());
    for (final String variable : variables) {
      idBuilder.append('[').append(variable).append("=${").append(variable).append("}]");
    }
    final ValueExpression bucketIdExpr = new ValueExpression(idBuilder.toString());
    return bucketIdExpr.evaluate(jobParameters).toString();
  }

  /**
   * @return The variables needed for bucket id creation, sorted by name.
   */
  private Collection<String> getBucketIdVariables() {
    final Set<String> variables = new TreeSet<String>();
    for (final String variable : _dot.getStore(_bucketMode).getVariables()) {
      if (!variable.startsWith(JobManagerConstants.SYSTEM_PARAMETER_PREFIX)) {
        variables.add(variable);
      }
    }
    for (final String variable : _dot.getObject(_bucketMode).getVariables()) {
      if (!variable.startsWith(JobManagerConstants.SYSTEM_PARAMETER_PREFIX)) {
        variables.add(variable);
      }
    }
    return variables;
  }

  /**
   * Creates and returns a BulkInfo pointing to the store object.
   * 
   * @return The BulkInfo object pointing to the objects in the store.
   */
  public BulkInfo createDataObject() {
    return createDataObject(null);
  }

  /**
   * Creates and returns a BulkInfo pointing to the store object.
   * 
   * @param variableValues
   *          values for data object name variables extracted from input object(s). If it contains no value for variable
   *          "_uuid" , a new value is created.
   * 
   * @return The BulkInfo object pointing to the objects in the store.
   */
  public BulkInfo createDataObject(final AnyMap variableValues) {
    final AnyMap parameters = DataFactory.DEFAULT.createAnyMap();
    if (variableValues != null) {
      parameters.putAll(variableValues);
    }
    if (_bucketParameters != null) {
      parameters.putAll(_bucketParameters);
    }
    if (!parameters.containsKey("_uuid")) {
      parameters.put("_uuid", IdGenerator.createBulkId());
    }
    // remove all unresolved variables from the object id:
    for (final String variable : _dot.getObject(_bucketMode).getVariables()) {
      if (!parameters.containsKey(variable)) {
        parameters.put(variable, "");
      }
    }
    final String objectName = _dot.getObject(_bucketMode).evaluate(parameters).getExpression();
    return new BulkInfo(_bucketDef.getName(), _storeName, objectName);
  }

  /**
   * @return the prefix of the data object names for this bucket.
   */
  public String getDataObjectNamePrefix() {
    final String objectName = _dot.getObject(_bucketMode).evaluate(_bucketParameters).getExpression();
    final int indexOfFirstVariable = objectName.indexOf("$");
    if (indexOfFirstVariable > 0) {
      final String objectNamePrefix = objectName.substring(0, indexOfFirstVariable);
      return objectNamePrefix;
    } else {
      return null;
    }
  }

  /**
   * @param objectId
   *          an object name in this bucket.
   * @return values for the variables in the object name template.
   */
  public Map<String, String> getDataObjectVariableValues(final String objectId) {
    return _dot.getObject(_bucketMode).extractValues(objectId);
  }
}
