/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.httphandler;

import java.net.MalformedURLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;
import org.eclipse.smila.http.server.util.URLCreator;
import org.eclipse.smila.jobmanager.DefinitionBase;
import org.eclipse.smila.jobmanager.InvalidConfigException;
import org.eclipse.smila.jobmanager.JobDefinition;
import org.eclipse.smila.jobmanager.JobManager;
import org.eclipse.smila.jobmanager.JobManagerConstants;
import org.eclipse.smila.jobmanager.JobManagerException;
import org.eclipse.smila.jobmanager.JobRunInfo;
import org.eclipse.smila.jobmanager.persistence.DefinitionPersistence;
import org.eclipse.smila.jobmanager.persistence.PersistenceException;
import org.eclipse.smila.utils.http.NotFoundHTTPResult;
import org.osgi.service.component.ComponentContext;

/**
 * Implements the handling of HTTP requests to request the job list or to define new jobs. <br>
 * 
 * URL pattern: <code>smila/jobmanager/jobs/</code> <br>
 * 
 * Methods allowed: <code>GET, POST</code>
 */
public class JobsHandler extends JsonRequestHandler {

  /** The key for the details. */
  public static final String KEY_RETURN_DETAILS = "returnDetails";

  /** job run data of the most current job run. */
  public static final String KEY_LATEST_JOB_RUN = "latestJobRun";

  /** The reference to the jobManager service. */
  private JobManager _jobManager;

  /** wrapper for definition storage adding access to definitions in config area. */
  private DefinitionPersistence _defPersistence;

  /**
   * {@inheritDoc}
   */
  @Override
  protected void activate(final ComponentContext context) {
    super.activate(context);
    _defPersistence = _jobManager.getDefinitionPersistence();
  }

  /**
   * clean up JobManager on service shutdown.
   * 
   * @param context
   *          OSGi service context.
   */
  protected void deactivate(final ComponentContext context) {
    _defPersistence = null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    if ("GET".equals(method)) {
      final AnyMap resultMap = DataFactory.DEFAULT.createAnyMap();
      final AnySeq jobList = DataFactory.DEFAULT.createAnySeq();
      final boolean showDetails;
      if (inputRecord != null && inputRecord.getMetadata().containsKey(KEY_RETURN_DETAILS)) {
        showDetails = inputRecord.getMetadata().getBooleanValue(KEY_RETURN_DETAILS).booleanValue();
      } else {
        showDetails = true;
      }
      for (final String jobName : _defPersistence.getJobs()) {
        final AnyMap map = DataFactory.DEFAULT.createAnyMap();
        map.put("name", jobName);
        map.put("url", URLCreator.create(getRequestHost(), requestUri, jobName).toString());
        if (showDetails) {
          final AnyMap jobDetails = getJobDetails(jobName, requestUri);
          if (!jobDetails.isEmpty()) {
            map.put(KEY_LATEST_JOB_RUN, jobDetails);
          }
        }
        jobList.add(map);
      }
      resultMap.put(JobDefinition.KEY_JOBS, jobList);
      return resultMap;
    } else { // POST
      if (inputRecord == null) {
        throw new IllegalArgumentException("Missing input, cannot create jobs from nothing.");
      }
      final AnyMap definition = inputRecord.getMetadata();
      // create timestamp for this workflow and store it with the data
      final Any timestamp = inputRecord.getFactory().createDateTimeValue(new Date());
      definition.put(DefinitionBase.KEY_TIMESTAMP, timestamp);
      // remove readOnly flag if present.
      definition.remove(DefinitionBase.KEY_READ_ONLY);
      final JobDefinition createdJob = JobDefinition.parseJob(definition);
      _defPersistence.addJob(createdJob);
      final AnyMap result = DataFactory.DEFAULT.createAnyMap();
      result.put("name", createdJob.getName());
      result.put(JobManagerConstants.TIMESTAMP, timestamp);
      result.put("url", URLCreator.create(getRequestHost(), requestUri, createdJob.getName()).toExternalForm());
      return result;
    }
  }

  /**
   * retrieves the job details.
   * <ul>
   * <li>status</li>
   * <li>startTime</li>
   * <li>endTime</li>
   * <li>successfulTaskCount</li>
   * <li>successfulWorkflowRunCount</li>
   * <li>failedAfterRetryTaskCount</li>
   * <li>failedWithoutRetryTaskCount</li>
   * <li>failedWorkflowRunCount</li>
   * </ul>
   * 
   * @param jobName
   *          the name of the job
   * @return a map containing key and value for the above keys and the given job.
   * @throws JobManagerException
   *           error while retrieving job run info.
   * @throws MalformedURLException
   *           cannot build the job run uri.
   */
  private AnyMap getJobDetails(final String jobName, final String requestUri) throws JobManagerException,
    MalformedURLException {
    final AnyMap map = DataFactory.DEFAULT.createAnyMap();
    final JobRunInfo currentRun = _jobManager.getJobRunInfo(jobName);
    String jobRunId = null;
    if (currentRun == null) {
      // no current run, get the latest job run id:
      final Collection<String> completedJobRunIds = _jobManager.getCompletedJobRunIds(jobName);
      if (!completedJobRunIds.isEmpty()) {
        // get the last one, it is the latest job run id
        final Iterator<String> iter = completedJobRunIds.iterator();
        while (iter.hasNext()) {
          jobRunId = iter.next();
        }
      }
    } else {
      jobRunId = currentRun.getId();
    }

    // did we get a job run id?
    if (jobRunId != null) {
      final AnyMap jobRunData = _jobManager.getJobRunData(jobName, jobRunId);
      safeCopyValues(jobRunData, map, Arrays.asList(JobManagerConstants.DATA_JOB_RUN_MODE,
        JobManagerConstants.DATA_JOB_STATE, JobManagerConstants.DATA_JOB_RUN_START_TIME,
        JobManagerConstants.DATA_JOB_RUN_END_TIME));
      if (jobRunData.containsKey(JobManagerConstants.WORKFLOW_RUN_COUNTER)) {
        final AnyMap workflowCounters = jobRunData.getMap(JobManagerConstants.WORKFLOW_RUN_COUNTER);
        safeCopyValues(workflowCounters, map, Arrays.asList(
          JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_WORKFLOW_RUNS,
          JobManagerConstants.DATA_JOB_NO_OF_FAILED_WORKFLOW_RUNS));
      }
      if (jobRunData.containsKey(JobManagerConstants.TASK_COUNTER)) {
        final AnyMap taskCounters = jobRunData.getMap(JobManagerConstants.TASK_COUNTER);
        safeCopyValues(taskCounters, map, Arrays.asList(JobManagerConstants.DATA_JOB_NO_OF_SUCCESSFUL_TASKS,
          JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_NOT_RETRIED,
          JobManagerConstants.DATA_JOB_NO_OF_FAILED_TASKS_RETRIED));
      }
      map.put("url", URLCreator.create(getRequestHost(), requestUri, jobName + "/" + jobRunId).toString());
    }
    return map;
  }

  /**
   * Copies a range of values from one map to another (if it existed in the source map in the first place.
   * 
   * @param sourceMap
   *          the source map
   * @param destinationMap
   *          the destination map
   * @param keys
   *          the keys which values are to be copied.
   */
  private void safeCopyValues(final AnyMap sourceMap, final AnyMap destinationMap, final Collection<String> keys) {
    for (final String key : keys) {
      if (sourceMap.containsKey(key)) {
        destinationMap.put(key, sourceMap.get(key));
      }
    }
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for InvalidConfigException and IllegalArgumentException as well as an
   * INTERNAL_SERVER_ERROR (500) for PersistenceException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof NotFoundHTTPResult) {
      // NotFoundHTTPResult before PersistenceException! (Because of ConfigNotFoundException.)
      return HttpStatus.NOT_FOUND;
    }
    if (ex instanceof PersistenceException) {
      return HttpStatus.INTERNAL_SERVER_ERROR;
    } else if (ex instanceof InvalidConfigException || ex instanceof IllegalArgumentException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * result status for POST requests should be CREATED.
   * 
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String method, final String requestUri) {
    if ("POST".equals(method)) {
      return HttpStatus.CREATED;
    }
    return super.getSuccessStatus(method, requestUri);
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return "GET".equals(method) || "POST".equals(method);
  }

  /**
   * Sets the reference to the JobManager service.
   * 
   * @param jobManager
   *          the reference to the JobManager service.
   */
  public void setJobManager(final JobManager jobManager) {
    _jobManager = jobManager;
  }

  /**
   * Resets the reference to the JobManager service to null if the given jobManager instance is set.
   * 
   * @param jobManager
   *          the reference to the current JobManager service.
   */
  public void unsetJobManager(final JobManager jobManager) {
    if (_jobManager == jobManager) {
      _jobManager = null;
    }
  }
}
