/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.internal;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.Any.ValueType;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.jobmanager.InvalidConfigException;
import org.eclipse.smila.jobmanager.util.ValueExpression;

/**
 * Some job manager related helper methods to deal with Any and their Exception handling.
 * 
 */
public final class AccessAny {

  /**
   * factory for Any.
   */
  public static final DataFactory FACTORY = DataFactory.DEFAULT;

  /** prevent instance creation. */
  private AccessAny() {
  }

  /**
   * get a field value from a Map and check that it has an expected type.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @param fieldType
   *          expected field type
   * @return the value for the key or null if it doesn't exist.
   * @throws InvalidConfigException
   *           any is not a map, value does not have expected type or error accessing any.
   */
  public static Any get(final AnyMap any, final String fieldName, final Any.ValueType fieldType)
    throws InvalidConfigException {
    Any field = null;
    field = any.get(fieldName);
    if (field != null && field.getValueType() != fieldType) {
      throw new InvalidConfigException("Field " + fieldName + " does not have type " + fieldType);
    }
    return field;
  }

  /**
   * get a value expression from a Map.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the value expression for the key or null if it doesn't exist.
   * @throws InvalidConfigException
   *           any is not a map, value is not a string or error accessing any.
   */
  public static ValueExpression getValueExpression(final AnyMap any, final String fieldName)
    throws InvalidConfigException {
    final String expression = getString(any, fieldName);
    if (expression != null) {
      return new ValueExpression(expression);
    }
    return null;
  }

  /**
   * get a string from a Map.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the string for the key or null if it doesn't exist.
   * @throws InvalidConfigException
   *           any is not a map, value is not a string or error accessing any.
   */
  public static String getString(final AnyMap any, final String fieldName) throws InvalidConfigException {
    final Value field = (Value) get(any, fieldName, ValueType.STRING);
    if (field != null) {
      return field.asString();
    }
    return null;
  }

  /**
   * get a sequence from a Map.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the sequence for the key or null if it doesn't exist.
   * @throws InvalidConfigException
   *           any is not a map, value is not a sequence or error accessing any.
   */
  public static AnySeq getSeq(final AnyMap any, final String fieldName) throws InvalidConfigException {
    final Any field = get(any, fieldName, ValueType.SEQ);
    if (field != null) {
      return field.asSeq();
    }
    return null;
  }

  /**
   * get a boolean from a Map. Accept the string values "true" or "false", too. 
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @param defaultValue
   *          value to return if key is not set.
   * @return the boolean for the key
   * @throws InvalidConfigException
   *           any is not a map, value is not a boolean or error accessing any.
   */
  public static boolean getBoolean(final AnyMap any, final String fieldName, final boolean defaultValue)
    throws InvalidConfigException {
    final Any field = any.get(fieldName);
    if (field == null) {
      return defaultValue;
    }
    if (field.isValue()) {
      try {
        return field.asValue().asBoolean();
      } catch (final InvalidValueTypeException ex) {
        throw new InvalidConfigException("Field " + fieldName + " does not have a boolean value.", ex);
      }
    }
    throw new InvalidConfigException("Field " + fieldName + " does not have a boolean value.");
  }

  /**
   * like {@link #getValueExpression(Any, String)}, but throw an exception if the field does not exists.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the value expression for the key.
   * @throws InvalidConfigException
   *           any is not a map, value does not exists or is not a string or error accessing any.
   */
  public static ValueExpression getValueExpressionRequired(final AnyMap any, final String fieldName)
    throws InvalidConfigException {
    return new ValueExpression(getStringRequired(any, fieldName));
  }

  /**
   * like {@link #getString(Any, String)}, but throw an exception if the field does not exists.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the string for the key.
   * @throws InvalidConfigException
   *           any is not a map, value does not exists or is not a string or error accessing any.
   */
  public static String getStringRequired(final AnyMap any, final String fieldName) throws InvalidConfigException {
    final String value = getString(any, fieldName);
    if (StringUtils.isEmpty(value)) {
      throw new InvalidConfigException("Missing field " + fieldName + " or empty value.");
    }
    return value;
  }

  /**
   * get list of strings from a field of the any.
   * 
   * @param any
   *          a map any.
   * @param fieldName
   *          key.
   * @return the list of strings or null if the field does not exists.
   * @throws InvalidConfigException
   *           any is not a map, value is not an sequence of strings or error accessing any.
   */
  public static List<String> getStringSeq(final AnyMap any, final String fieldName) throws InvalidConfigException {
    final AnySeq field = (AnySeq) get(any, fieldName, ValueType.SEQ);
    List<String> values = null;
    if (field != null) {
      values = new ArrayList<String>();
      try {
        for (final Any element : field) {
          if (!element.isString()) {
            throw new InvalidConfigException("Sequence in field " + fieldName + " must contain strings only");
          }
          values.add(((Value) element).asString());
        }
      } catch (final InvalidConfigException ex) {
        throw ex;
      }
    }
    return values;
  }
}
