/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.jobmanager.taskgenerator;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements a TaskGeneratorProvider for selecting TaskGenerator implementations.
 */
public class TaskGeneratorProviderImpl implements TaskGeneratorProvider {

  /** the TaskGenerators to use. */
  private final Map<String, TaskGenerator> _taskGenerators = new HashMap<String, TaskGenerator>();

  /** local logger. */
  private final Log _log = LogFactory.getLog(getClass());

  /** {@inheritDoc} */
  @Override
  public TaskGenerator getTaskGenerator(final String taskGeneratorName) {
    if (_taskGenerators.containsKey(taskGeneratorName)) {
      return _taskGenerators.get(taskGeneratorName);
    }
    throw new IllegalArgumentException("Could not find TaskGenerator for name '" + taskGeneratorName + "'");
  }

  /** {@inheritDoc} */
  @Override
  public TaskGenerator getDefaultTaskGenerator() {
    return getTaskGenerator(DEFAULT_TASK_GENERATOR);
  }

  /**
   * Add a TaskGenerator..
   * 
   * @param taskGenerator
   *          the new TaskGenerator
   */
  public synchronized void addTaskGenerator(final TaskGenerator taskGenerator) {
    _log.info("Have TaskGenerator '" + taskGenerator.getName() + "', class " + taskGenerator.getClass().getName());
    _taskGenerators.put(taskGenerator.getName(), taskGenerator);
  }

  /**
   * Remove the given TaskGenerator.
   * 
   * @param taskGenerator
   *          the TaskGenerator
   */
  public synchronized void removeTaskGenerator(final TaskGenerator taskGenerator) {
    if (_taskGenerators.values().contains(taskGenerator)) {
      _taskGenerators.remove(taskGenerator.getName());
    }
  }
}
