package org.eclipse.smila.processing.worker;

import java.io.IOException;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.smila.blackboard.Blackboard;
import org.eclipse.smila.blackboard.BlackboardAccessException;
import org.eclipse.smila.blackboard.BlackboardFactory;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.processing.ProcessingException;
import org.eclipse.smila.processing.util.ProcessingConstants;
import org.eclipse.smila.taskworker.TaskContext;
import org.eclipse.smila.taskworker.TaskLog;
import org.eclipse.smila.taskworker.Worker;
import org.eclipse.smila.taskworker.input.RecordInput;
import org.eclipse.smila.taskworker.output.RecordOutput;

/**
 * Common stuff for PipeletProcessorWorker and PipelineProcessorWorker.
 */
public abstract class ProcessingWorker implements Worker {

  /** the workers input slot name . */
  public static final String INPUT_SLOT_NAME = "input";

  /** the workers output slot name . */
  public static final String OUTPUT_SLOT_NAME = "output";

  /** flag if the blackboard warning has already been logged to prevent excessive logging. */
  protected static boolean s_noBinStorageAvailableLogged;

  /** protected log. */
  protected final Log _log = LogFactory.getLog(getClass());

  /** blackboard factory. Set by DS */
  protected BlackboardFactory _blackboardFactory;

  /** use bin storage? */
  protected boolean _tryToUseBinStorage = true;

  @Override
  public void perform(final TaskContext taskContext) throws Exception {
    final AnyMap parameters = getTaskParameters(taskContext);
    final RecordInput recordInput = taskContext.getInputs().getAsRecordInput(INPUT_SLOT_NAME);
    final RecordOutput recordOutput = taskContext.getOutputs().getAsRecordOutput(OUTPUT_SLOT_NAME);
    _tryToUseBinStorage = true;
    final boolean success = perform(parameters, recordInput, recordOutput, taskContext);
    if (!success) {
      throw new ProcessingException("None of the records of task " + taskContext.getTask()
        + " could be successfully processed, have a look at the log for details.");
    }
  }

  /**
   * 
   * @param parameters
   *          task parameters, converted to an AnyMap
   * @param recordInput
   *          input bulk
   * @param recordOutput
   *          output bulk, can be null
   * @param taskContext
   *          task context
   * @return true if at least one record was processed successfully.
   * @throws Exception
   *           bulk could not be processed
   */
  public abstract boolean perform(AnyMap parameters, RecordInput recordInput, RecordOutput recordOutput,
    TaskContext taskContext) throws Exception;

  /** create AnyMap with task parameters. */
  private AnyMap getTaskParameters(final TaskContext taskContext) {
    final AnyMap parameterMap = DataFactory.DEFAULT.cloneAnyMap(taskContext.getTask().getParameters());
    if (parameterMap.get(ProcessingConstants.KEY_FAIL_ON_ERROR) == null) {
      parameterMap.put(ProcessingConstants.KEY_FAIL_ON_ERROR, "false");
    }
    return parameterMap;
  }

  /**
   * append the resulting records to the bulk. Errors on blackboard access are catched and logged as warnings to the
   * task log, as they are considered as record-specific non-recoverable errors. In any case, the blackboard is emptied
   * afterwards.
   * @param recordOutput
   *          where to write the records. Can be null (is optional in worker description)
   */
  protected void writeResultRecords(final Blackboard blackboard, final String[] resultIds,
    final RecordOutput recordOutput, final TaskLog taskLog) throws ObjectStoreException, IOException {
    try {
      if (resultIds != null && recordOutput != null) {
        for (final String resultId : resultIds) {
          try {
            recordOutput.writeRecord(blackboard.getRecord(resultId));
          } catch (final BlackboardAccessException ex) {
            taskLog.warn("Failed to read result record " + resultId + " from blackboard, skipping it.", ex);
          }
        }
      }
    } finally {
      blackboard.unload();
    }
  }

  /**
   * Creates a blackboard.
   */
  protected Blackboard getBlackboard() throws Exception {
    Blackboard blackboard;
    try {
      blackboard = _blackboardFactory.createBlackboard(false, _tryToUseBinStorage);
    } catch (final BlackboardAccessException e) {
      _tryToUseBinStorage = false;
      if (!s_noBinStorageAvailableLogged) {
        _log.warn("Could not get a blackboard with binary storage, using a transient one. "
          + "Attachments won't be accessible by the worker.");
        s_noBinStorageAvailableLogged = true;
      }
      blackboard = _blackboardFactory.createTransientBlackboard();
    }
    return blackboard;
  }

  /** set blackboard factory reference (used by DS). */
  public void setBlackboardFactory(final BlackboardFactory factory) {
    _blackboardFactory = factory;
  }

  /** remove blackboard factory reference (used by DS). */
  public void unsetBlackboardFactory(final BlackboardFactory factory) {
    if (_blackboardFactory == factory) {
      _blackboardFactory = null;
    }
  }
}
