/**********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskworker.input;

import java.io.IOException;
import java.util.Map;

import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcSerializationUtils;
import org.eclipse.smila.ipc.IpcFactory;
import org.eclipse.smila.ipc.IpcStreamReader;
import org.eclipse.smila.ipc.bon.BinaryFactory;
import org.eclipse.smila.objectstore.BadRequestException;
import org.eclipse.smila.objectstore.ObjectStoreException;
import org.eclipse.smila.objectstore.ObjectStoreService;

import org.eclipse.smila.taskmanager.BulkInfo;
import org.eclipse.smila.taskworker.Counters;

/**
 * Provides access to objects like record bulks that are sequences of BON {@link Record}s. You can get single
 * {@link Record} from the objects, one at a time, by calling the {@link #getRecord()} method. When end-of-stream is
 * reached, null is returned. You can also access the {@link IpcStreamReader} BON parser directly in case you do not
 * want to read complete records at once. However, you should not mix up {@link #getRecord()} calls with calls to the
 * {@link IpcStreamReader} as your direct calls will probably confuse the record parsing.
 * 
 */
public class RecordInput extends Input {
  /** BON IPC factory. */
  private static final IpcFactory IPCFACTORY = new BinaryFactory();

  /** used to access the objectstore input stream. */
  private final StreamInput _streamInput;

  /** BON parser. */
  private IpcStreamReader _reader;

  /** BON-2-record translation. */
  private final IpcSerializationUtils _serializationUtils = new IpcSerializationUtils();

  /** number of parsed records. */
  private long _recordCount;

  /** create instance. */
  public RecordInput(final BulkInfo dataObject, final ObjectStoreService objectStore) {
    super(dataObject, objectStore);
    _streamInput = new StreamInput(dataObject, objectStore);
  }

  /**
   * @return next record on stream, or null, if end-of-stream is reached.
   */
  public Record getRecord() throws ObjectStoreException, IOException {
    final IpcStreamReader reader = getStreamReader();
    final long start = startTime();
    try {
      final Record record = _serializationUtils.stream2record(reader);
      if (record != null) {
        _recordCount++;
      }
      return record;
    } catch (final IllegalStateException ex) {
      throw new BadRequestException("Error parsing BON stream to record", ex);
    } finally {
      timePerform(start);
    }
  }

  /**
   * @return BON parser on data object stream. Records are not counted when this is used by worker immediately.
   */
  public IpcStreamReader getStreamReader() throws ObjectStoreException, IOException {
    if (_reader == null) {
      final long start = startTime();
      _reader = IPCFACTORY.newStreamReader(_streamInput.getStream());
      timeOpen(start);
    }
    return _reader;
  }

  /** @return number of records parsed so far using {@link #getRecord()} */
  public long getRecordCount() {
    return _recordCount;
  }

  /**
   * @see StreamInput#getBytesRead()
   */
  public long getBytesRead() {
    return _streamInput.getBytesRead();
  }

  /** @return counter map of superclass extended by the number of bytes and number of records read from the stream. */
  @Override
  public Map<String, Number> getCounter() {
    final Map<String, Number> counter = _streamInput.getCounter();
    Counters.add(counter, "recordCount", _recordCount);
    return counter;
  }

  /** close BON reader, if open. */
  @Override
  public void close() {
    if (_reader != null) {
      try {
        _reader.closeWithoutStream();
      } catch (final IOException ex) {
        ; // ignore.
      }
    }
    _streamInput.close();
  }

  @Override
  public String toString() {
    return "RecordInput: store = " + getStoreName() + ", object = " + getObjectName();
  }

}
