/*******************************************************************************
 * Copyright (c) 2008 empolis GmbH and brox IT Solutions GmbH. All rights reserved. This program and the accompanying
 * materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Thomas Menzel (brox IT Solution GmbH) - initial creator
 *******************************************************************************/

package org.eclipse.smila.search.api.helper;

import org.apache.commons.lang.NullArgumentException;
import org.eclipse.smila.datamodel.Any;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.Value;
import org.eclipse.smila.search.api.SearchResultConstants;

/**
 * wrapper helper class to build the result record. This is intentionally a subclass of the {@link ResultAccessor} as to
 * also provide read access to the props that are set here.
 */
public class ResultBuilder extends ResultAccessor {

  /**
   * Creates a new builder and adds an AnySeq for the {@link SearchResultConstants#RECORDS}.
   * 
   * @param workflowName
   * @param result
   */
  public ResultBuilder(final String workflowName, final Record result) {
    super(workflowName, result);

    _result.getMetadata().getSeq(SearchResultConstants.RECORDS, true);
  }

  /**
   * Adds a new result item to the list. the given args are mandatory and will be set at the item.
   * 
   * @return the map containing the id and score/weight under their proper names.
   */
  public AnyMap addResultItem(final String id, final Double score) {
    final AnyMap resultItem = getResultRecords().getFactory().createAnyMap();
    getResultRecords().add(resultItem);
    resultItem.put(Record.RECORD_ID, id);
    resultItem.put(SearchResultConstants.WEIGHT, score);
    return resultItem;
  }

  /**
   * sets value for {@link SearchResultConstants#COUNT}.
   */
  public void setCount(final Long count) {
    _result.getMetadata().put(SearchResultConstants.COUNT, count);
  }

  /**
   * sets value for {@link SearchResultConstants#RUNTIME}.
   */
  public void setRuntime(final Long millis) {
    _result.getMetadata().put(SearchResultConstants.RUNTIME, millis);
  }

  /**
   * Adds group Seq with the given name and returns it. The groups map is created on the fly if needs be.
   */
  public AnySeq addGroup(final String groupName) {
    final AnyMap groups = _result.getMetadata().getMap(SearchResultConstants.GROUPS, true);

    return groups.getSeq(groupName, true);
  }

  /**
   * Adds the map containing the count and value for the group value.
   * 
   * @param group
   *          NullArgumentException if null
   * @param value
   *          NullArgumentException if null
   * @param count
   *          if null the count property is not set
   * @return the map of the added group
   */
  public AnyMap addGroupValue(final AnySeq group, final Any value, final Long count) {
    if (group == null) {
      throw new NullArgumentException("group must not be null");
    }
    if (value == null) {
      throw new NullArgumentException("value must not be null");
    }

    final AnyMap groupValue = group.getFactory().createAnyMap();
    groupValue.put(SearchResultConstants.VALUE, value);
    if (count != null) {
      groupValue.put(SearchResultConstants.COUNT, count);
    }
    group.add(groupValue);
    return groupValue;
  }

  /**
   * Adds the map containing the count and value for the group value.
   * 
   * @param group
   *          any valid literal object that can be passed into
   *          {@link org.eclipse.smila.datamodel.DataFactory#autoConvertValue(Object)}
   * @return the map for added value containing the value and count
   */
  public AnyMap addGroupValue(final AnySeq group, final Object value, final Long count) {
    final Value autoConvertValue = group.getFactory().autoConvertValue(value);
    return addGroupValue(group, autoConvertValue, count);
  }

}
