/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Andreas Weber (Attensity Europe GmbH) - initial implementation
 **********************************************************************************************************************/

package org.eclipse.smila.taskmanager.httphandler;

import java.util.Collection;
import java.util.List;

import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.InvalidValueTypeException;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.json.JsonRequestHandler;

import org.eclipse.smila.taskmanager.BadParameterTaskmanagerException;
import org.eclipse.smila.taskmanager.Task;
import org.eclipse.smila.taskmanager.TaskManager;

/**
 * Implements the handling of HTTP requests of a worker to obtain a task.
 * 
 * URL pattern: taskmanager/-workername-/
 */
public class WorkerTaskHandler extends JsonRequestHandler {

  /** key of workerHost property. */
  public static final String KEY_WORKERHOST = "workerHost";

  /** the qualifier key. */
  public static final String KEY_QUALIFIER = "qualifier";

  /** The reference to the TaskManager service. */
  private TaskManager _taskManager;

  /**
   * {@inheritDoc}
   */
  @Override
  public Object process(final String method, final String requestUri, final Record inputRecord) throws Exception {
    String workerHost = null;
    Collection<String> qualifiers = null;
    if (inputRecord != null) {
      workerHost = getWorkerHostName(inputRecord);
      qualifiers = getQualifier(inputRecord);
    }
    final String workerName = getWorkerName(requestUri);
    final Task task = _taskManager.getTask(workerName, workerHost, qualifiers);
    if (task != null) {
      return task.toAny();
    }
    return null;
  }

  /**
   * @return worker's hostname specified in input record.
   */
  private String getWorkerHostName(final Record inputRecord) {
    String workerHost = null;
    final AnyMap inputRecordAsAny = inputRecord.getMetadata();
    if (inputRecordAsAny.containsKey(KEY_WORKERHOST)) {
      workerHost = (inputRecordAsAny.getStringValue(KEY_WORKERHOST));
    }
    return workerHost;
  }

  /**
   * @return task qualifier, if specified in input records.
   */
  protected Collection<String> getQualifier(final Record inputRecord) {
    Collection<String> qualifier = null;
    try {
      final AnyMap inputRecordAsAny = inputRecord.getMetadata();
      if (inputRecordAsAny.containsKey(KEY_QUALIFIER)) {
        qualifier = inputRecordAsAny.getSeq(KEY_QUALIFIER).asStrings();
      }
    } catch (final Exception ex) {
      throw new IllegalArgumentException("Cannot convert qualifiers.", ex);
    }
    return qualifier;
  }

  /**
   * Adds HTTP result code 400 ("BAD_REQUEST") for IllegalArgumentException to the exception handling of
   * {@link JsonRequestHandler#getErrorStatus(String, String, Throwable)}. <br>
   * 
   * @param method
   *          HTTP method
   * @param requestUri
   *          request URI
   * @param ex
   *          an exception
   * @return error status code.
   */
  @Override
  protected int getErrorStatus(final String method, final String requestUri, final Throwable ex) {
    if (ex instanceof IllegalArgumentException || ex instanceof InvalidValueTypeException
      || ex instanceof BadParameterTaskmanagerException) {
      return HttpStatus.BAD_REQUEST;
    }
    return super.getErrorStatus(method, requestUri, ex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected int getSuccessStatus(final String requestMethod, final String requestUri, final Record inputRecord,
    final Object resultObject) {
    if (resultObject == null) {
      return HttpStatus.NO_CONTENT;
    }
    return super.getSuccessStatus(requestMethod, requestUri, inputRecord, resultObject);
  }

  /**
   * Gets the job name from the requestUri.
   * 
   * @param requestUri
   *          the request URI.
   * @return he job name from the requestUri.
   */
  private String getWorkerName(final String requestUri) {
    final List<String> uriParts = getDynamicUriParts(requestUri);
    if (uriParts.size() > 0) {
      return uriParts.get(0);
    }
    throw new IllegalArgumentException("Invalid request URI, does not match uri pattern '" + getUriPattern() + "'.");
  }

  /**
   * {@inheritDoc}
   * 
   * GET and POST are currently the only valid methods.
   */
  @Override
  protected boolean isValidMethod(final String method, final String requestUri) {
    return ("GET".equals(method) || "POST".equals(method));
  }

  /**
   * @param taskManager
   *          the reference to the JobManager service.
   */
  public void setTaskManager(final TaskManager taskManager) {
    _taskManager = taskManager;
  }

  /**
   * @param taskManager
   *          the reference to the current JobManager service.
   */
  public void unsetTaskManager(final TaskManager taskManager) {
    if (_taskManager == taskManager) {
      _taskManager = null;
    }
  }

}
