/*******************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Juergen Schumacher, Andreas Weber, Drazen Cindric, Andreas Schank (all Attensity Europe GmbH) - initial
 * implementation
 **********************************************************************************************************************/
package org.eclipse.smila.bulkbuilder.test;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import junit.framework.TestCase;

import org.eclipse.smila.bulkbuilder.BulkbuilderException;
import org.eclipse.smila.bulkbuilder.helper.MicroBulkbuilder;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;

/** Tests for the MicroBulkBuilder class. */
public class TestMicroBulkbuilder extends TestCase {

  private static final Record RECORD = DataFactory.DEFAULT.createRecord("myId");

  /** tests that exceeding the limit for parallel micro bulk processing will cause an error. */
  public void testExceedingParallelLimit() throws Exception {
    final int limit = 10;
    final MicroBulkbuilder microBulkBuilder = new MicroBulkbuilder(limit);

    final ExecutorService executor = Executors.newFixedThreadPool(limit);
    final CompletionService<byte[]> cs = new ExecutorCompletionService<byte[]>(executor);
    final List<TestCallable> testList = new ArrayList<TestCallable>();
    try {
      // create parallel threads until limit is reached
      for (int i = 0; i < limit; i++) {
        final TestCallable t = new TestCallable(microBulkBuilder);
        testList.add(t);
        cs.submit(t);
      }
      Thread.sleep(500);
      // ... so next thread should fail
      final String microBulkId = UUID.randomUUID().toString();
      try {
        microBulkBuilder.addToMicroBulk(microBulkId, RECORD);
        fail("Expected exception because limit of parallel micro bulk processing is exceeded.");
      } catch (final BulkbuilderException e) {
        ; // expected
      }
      // ... but the others should have no errors
      for (int i = 0; i < limit; i++) {
        final TestCallable t = testList.get(i);
        t.finish();
        final byte[] microBulk = cs.take().get();
        assertTrue("Should return micro bulk", microBulk != null && microBulk.length > 0);
      }
      // ... and now we should be able to add microbulks again
      microBulkBuilder.addToMicroBulk(microBulkId, RECORD);
    } finally {
      executor.shutdownNow();
    }
  }

  /** test helper class. */
  private class TestCallable implements Callable<byte[]> {

    private final MicroBulkbuilder _microBulkBuilder;

    private final String _microBulkId = UUID.randomUUID().toString();

    private boolean _finished;

    /** */
    public TestCallable(final MicroBulkbuilder microBulkBuilder) {
      _microBulkBuilder = microBulkBuilder;
    }

    @Override
    public byte[] call() throws Exception {
      while (!_finished) {
        _microBulkBuilder.addToMicroBulk(_microBulkId, RECORD);
        Thread.sleep(10);
      }
      return _microBulkBuilder.finishMicroBulk(_microBulkId);
    }

    /** finish the execution. */
    public void finish() {
      _finished = true;
    }

  }

}
