/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation Andreas Weber (Attensity Europe GmbH) -
 * changes for job management
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.test;

import java.util.Collection;
import java.util.Map;

import org.eclipse.smila.connectivity.deltaindexing.DeltaIndexingManager;
import org.eclipse.smila.connectivity.framework.CrawlState;
import org.eclipse.smila.connectivity.framework.CrawlerController;
import org.eclipse.smila.connectivity.framework.util.CrawlThreadState;

/**
 * The Class TestCrawlerController.
 */
public class TestCrawlerController extends ConnectivityTestBase {

  /** a valid data source ID. */
  public static final String CRAWLER_DATASOURCEID = "FilesystemConfigExample.xml";

  /**
   * Constant for thread sleep time.
   */
  private static final int SLEEP_TIME = 3000;

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
    super.setUp();
  }

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#tearDown()
   */
  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
  }

  /**
   * Test {@link CrawlerController#hasActiveCrawls()}.
   * 
   * @throws Exception
   *           a Exception
   */
  public void testHasActiveCrawls() throws Exception {
    final CrawlerController crawlerController = getService(CrawlerController.class);

    final boolean hasActiveCrawls = crawlerController.hasActiveCrawls();
    assertFalse(hasActiveCrawls);
  }

  /**
   * Test {@link CrawlerController#getAgentTasksState()}.
   * 
   * @throws Exception
   *           a Exception
   */
  public void testGetCrawlerTasksState() throws Exception {
    final CrawlerController crawlerController = getService(CrawlerController.class);

    final Map<String, CrawlState> crawlerTasksState = crawlerController.getCrawlerTasksState();
    assertNotNull(crawlerTasksState);
    assertEquals(0, crawlerTasksState.size());
  }

  /**
   * Test {@link CrawlerController#getAvailableConfigurations()}.
   * 
   * @throws Exception
   *           a Exception
   */
  public void testGetAvailableConfigurations() throws Exception {
    final CrawlerController crawlerController = getService(CrawlerController.class);

    final Collection<String> configs = crawlerController.getAvailableConfigurations();
    assertNotNull(configs);
    assertEquals(3, configs.size());
  }

  /**
   * Test a Crawl run.
   * 
   * @throws Exception
   *           the Exception
   */
  public void testStartCrawl() throws Exception {
    final String jobName = getClass().getSimpleName() + "-testStartCrawl";
    final String jobRunId = startJob(jobName);
    try {
      testStartCrawl(CRAWLER_DATASOURCEID, jobName, false);
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test a Crawl run.
   * 
   * @throws Exception
   *           the Exception
   */
  public void testStopCrawl() throws Exception {
    final String jobName = getClass().getSimpleName() + "-testStopCrawl";
    final String jobRunId = startJob(jobName);
    try {
      testStartCrawl(CRAWLER_DATASOURCEID, jobName, true);
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test a Crawl run.
   * 
   * @throws Exception
   *           the Exception
   */
  public void testStartCrawlCompounds() throws Exception {
    final String dataSourceId = "ZipConfigExample.xml";
    final String jobName = getClass().getSimpleName() + "-testStartCrawlCompounds";
    final String jobRunId = startJob(jobName);
    try {
      testStartCrawl(dataSourceId, jobName, false);
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test a Crawl run.
   * 
   * @param dataSourceId
   *          the id of the data source
   * @param jobName
   *          name of the job
   * @param interrupt
   *          the interrupt
   * 
   * @throws Exception
   *           the Exception
   */
  private void testStartCrawl(final String dataSourceId, final String jobName, final boolean interrupt)
    throws Exception {
    final DeltaIndexingManager deltaIndexingManager = getService(DeltaIndexingManager.class);
    deltaIndexingManager.unlockDatasources();

    final CrawlerController crawlerController = getService(CrawlerController.class);
    crawlerController.startCrawl(dataSourceId, jobName);

    boolean hasActiveCrawls = crawlerController.hasActiveCrawls();
    assertTrue(hasActiveCrawls);

    Map<String, CrawlState> crawlStates = crawlerController.getCrawlerTasksState();
    assertNotNull(crawlStates);
    CrawlState crawlState = crawlStates.get(dataSourceId);
    assertNotNull(crawlState);
    assertEquals(CrawlThreadState.Running, crawlState.getState());
    assertNull(crawlState.getLastError());

    if (interrupt) {
      // stop crawler
      crawlerController.stopCrawl(dataSourceId);
      Thread.sleep(SLEEP_TIME);
    } else {
      // wait for crawl to finish
      Thread.sleep(SLEEP_TIME);
      while (CrawlThreadState.Running.equals(crawlStates.get(dataSourceId).getState())) {
        Thread.sleep(SLEEP_TIME);
        crawlStates = crawlerController.getCrawlerTasksState();
      }
    }

    hasActiveCrawls = crawlerController.hasActiveCrawls();
    assertFalse(hasActiveCrawls);

    crawlStates = crawlerController.getCrawlerTasksState();
    assertNotNull(crawlStates);
    crawlState = crawlStates.get(dataSourceId);
    assertNotNull(crawlState);
    if (interrupt) {
      assertEquals(CrawlThreadState.Stopped, crawlState.getState());
    } else {
      assertEquals(CrawlThreadState.Finished, crawlState.getState());
    }
    assertNull(crawlState.getLastError());
  }
}
