/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Daniel Stucky (empolis GmbH) - initial API and implementation Sebastian Voigt (brox IT Solutions GmbH)
 * Andreas Weber (Attensity Europe GmbH) - changes for job management
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.test;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.eclipse.smila.connectivity.framework.CrawlerControllerAgent;
import org.eclipse.smila.connectivity.framework.util.CrawlThreadState;

/**
 * The Class CrawlerControllerAgent.
 */
public class TestCrawlerControllerAgent extends ConnectivityTestBase {

  /**
   * Constant for thread sleep time.
   */
  private static final int SLEEP_TIME = 3000;

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#setUp()
   */
  @Override
  protected void setUp() throws Exception {
  }

  /**
   * {@inheritDoc}
   * 
   * @see junit.framework.TestCase#tearDown()
   */
  @Override
  protected void tearDown() throws Exception {
  }

  /**
   * Test error messages.
   * 
   * @throws Exception
   *           a Exception
   */
  public void testErrorMessages() throws Exception {
    final CrawlerControllerAgent cca = getService(CrawlerControllerAgent.class);
    assertNotNull(cca);

    final String dataSourceId = "dummy";
    final String jobName = "dummyJob";

    String msg = cca.startCrawlerTask(dataSourceId, jobName);
    assertNotNull(msg);
    assertTrue(msg.contains("org.eclipse.smila.utils.config.ConfigurationLoadException"));
    assertTrue(msg.contains(dataSourceId + ".xml"));
    try {
      msg = cca.startCrawlerTask(null, jobName);
      fail("Expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      ; // expected
    }

    msg = cca.stopCrawlerTask(dataSourceId);
    assertNotNull(msg);
    assertEquals("Could not stop crawl for DataSourceId '" + dataSourceId + "'. No CrawlThread exists.", msg);

    try {
      msg = cca.stopCrawlerTask(null);
      fail("Expected IllegalArgumentException");
    } catch (final IllegalArgumentException e) {
      ; // expected
    }
  }

  /**
   * Test {@link CrawlerControllerAgent#getActiveAgentTaskStatus()}.
   * 
   * @throws Exception
   *           a Exception
   */
  public void testGetActiveCrawlerTaskStatus() throws Exception {
    final CrawlerControllerAgent cca = getService(CrawlerControllerAgent.class);
    assertNotNull(cca);
    final String jobName = getClass().getSimpleName() + "-testGetActiveCrawlerTaskStatus";
    final String jobRunId = startJob(jobName);

    cca.startCrawlerTask(TestCrawlerController.CRAWLER_DATASOURCEID, jobName);
    Map<String, String> crawlerTasksState = cca.getCrawlerTasksState();
    assertNotNull(crawlerTasksState);
    assertEquals(CrawlThreadState.Running.name(), crawlerTasksState.get(TestCrawlerController.CRAWLER_DATASOURCEID));

    cca.stopCrawlerTask(TestCrawlerController.CRAWLER_DATASOURCEID);
    while (CrawlThreadState.Running.name()
      .equals(crawlerTasksState.get(TestCrawlerController.CRAWLER_DATASOURCEID))) {
      crawlerTasksState = cca.getCrawlerTasksState();
      Thread.sleep(SLEEP_TIME);
    }

    assertNotNull(crawlerTasksState);
    assertEquals(CrawlThreadState.Stopped.name(), crawlerTasksState.get(TestCrawlerController.CRAWLER_DATASOURCEID));
    finishJob(jobName, jobRunId);
  }

  /**
   * Test {@link CrawlerControllerAgent#getAvailableCrawlerTasks()}.
   * 
   * @throws Exception
   *           a Exception
   */
  public void testGetAvailableConfigurations() throws Exception {
    final CrawlerControllerAgent cca = getService(CrawlerControllerAgent.class);
    assertNotNull(cca);

    final String[] configs = cca.getAvailableCrawlerTasks();
    assertNotNull(configs);
    assertEquals(3, configs.length);
  }

  /**
   * Test a Crawl run.
   * 
   * @throws Exception
   *           the Exception
   */
  public void testStartCrawl() throws Exception {
    final String jobName = getClass().getSimpleName() + "-testStartCrawl";
    final String jobRunId = startJob(jobName);
    try {
      testStartCrawl(false, jobName);
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test a Crawl run.
   * 
   * @throws Exception
   *           the Exception
   */
  public void testStopCrawl() throws Exception {
    final String jobName = getClass().getSimpleName() + "-testStopCrawl";
    final String jobRunId = startJob(jobName);
    try {
      testStartCrawl(true, jobName);
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test a Crawl run.
   * 
   * @param interrupt
   *          the interrupt
   * 
   * @throws Exception
   *           the Exception
   */
  private void testStartCrawl(final boolean interrupt, final String jobName) throws Exception {
    final CrawlerControllerAgent cca = getService(CrawlerControllerAgent.class);
    assertNotNull(cca);

    cca.startCrawlerTask(TestCrawlerController.CRAWLER_DATASOURCEID, jobName);

    Map<String, String> crawlStates = cca.getCrawlerTasksState();
    assertNotNull(crawlStates);
    assertEquals(CrawlThreadState.Running.toString(), crawlStates.get(TestCrawlerController.CRAWLER_DATASOURCEID));

    if (interrupt) {
      // stop crawler
      cca.stopCrawlerTask(TestCrawlerController.CRAWLER_DATASOURCEID);
      Thread.sleep(SLEEP_TIME);
    } else {
      // wait for crawl to finish
      Thread.sleep(SLEEP_TIME);
      while (CrawlThreadState.Running.toString()
        .equals(crawlStates.get(TestCrawlerController.CRAWLER_DATASOURCEID))) {
        Thread.sleep(SLEEP_TIME);
        crawlStates = cca.getCrawlerTasksState();
      }
    }

    crawlStates = cca.getCrawlerTasksState();
    assertNotNull(crawlStates);
    if (interrupt) {
      assertEquals(CrawlThreadState.Stopped.toString(), crawlStates.get(TestCrawlerController.CRAWLER_DATASOURCEID));
    } else {
      assertEquals(CrawlThreadState.Finished.toString(),
        crawlStates.get(TestCrawlerController.CRAWLER_DATASOURCEID));
    }
  }

  /**
   * test the Function get the known Crawlers.
   * 
   * @throws Exception
   *           in case of weird problems
   */
  public void testAvailableCrawlers() throws Exception {
    final CrawlerControllerAgent cca = getService(CrawlerControllerAgent.class);
    final String[] availCrawlers = cca.getAvailableCrawlers();
    assert availCrawlers.length == 2;
    final List<String> a = Arrays.asList(availCrawlers);
    assert a.indexOf("FileSystemCrawler") > 0;
    assert a.indexOf("WebCrawler") > 0;
  }

}
