/***********************************************************************************************************************
 * Copyright (c) 2008, 2011 Attensity Europe GmbH and brox IT Solutions GmbH. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: Drazen cindric (Attensity Europe GmbH) - changes for job management
 **********************************************************************************************************************/
package org.eclipse.smila.connectivity.framework.test;

import java.util.Map;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.RequestEntity;
import org.apache.commons.httpclient.methods.StringRequestEntity;
import org.eclipse.smila.connectivity.deltaindexing.DeltaIndexingManager;
import org.eclipse.smila.connectivity.framework.AgentController;
import org.eclipse.smila.connectivity.framework.AgentState;
import org.eclipse.smila.connectivity.framework.CrawlState;
import org.eclipse.smila.connectivity.framework.CrawlerController;
import org.eclipse.smila.connectivity.framework.httphandler.AgentControlHandler;
import org.eclipse.smila.connectivity.framework.httphandler.AgentHandler;
import org.eclipse.smila.connectivity.framework.httphandler.AgentsHandler;
import org.eclipse.smila.connectivity.framework.httphandler.CrawlerControlHandler;
import org.eclipse.smila.connectivity.framework.httphandler.CrawlerHandler;
import org.eclipse.smila.connectivity.framework.httphandler.CrawlersHandler;
import org.eclipse.smila.connectivity.framework.util.AgentThreadState;
import org.eclipse.smila.connectivity.framework.util.CrawlThreadState;
import org.eclipse.smila.datamodel.AnyMap;
import org.eclipse.smila.datamodel.AnySeq;
import org.eclipse.smila.datamodel.DataFactory;
import org.eclipse.smila.datamodel.Record;
import org.eclipse.smila.datamodel.ipc.IpcRecordWriter;
import org.eclipse.smila.http.server.HttpStatus;
import org.eclipse.smila.http.server.util.RequestHandler;

/** Test class for the handlers. */
public class TestHandler extends ConnectivityTestBase {

  /** Constant for thread sleep time. */
  private static final int SLEEP_TIME = 3000;

  /** a valid data source ID for crawler. */
  private static final String CRAWLER_DATASOURCEID = "FilesystemConfigExample.xml";

  /** data source for waiting crawler. */
  private static final String WAIT_CRAWLER_DATASOURCEID = "TestWaitCrawlerConfig.xml";

  /** a valid data source ID for agent. */
  private static final String AGENT_DATASOURCEID = "mockAgent";

  /** The pattern for the CrawlersHandler. */
  private static final String PATTERN_CRAWLERS = "/crawlers/?$";

  /** The pattern for the CrawlerHandler. */
  private static final String PATTERN_CRAWLER = "/crawlers/\\([^/]+\\)/?$";

  /** The pattern for the CrawlerControlHandler. */
  private static final String PATTERN_CRAWLER_CONTROL = "/crawlers/\\([^/]+\\)/finish/?$";

  /** The pattern for the AgentsHandler. */
  private static final String PATTERN_AGENTS = "/agents/?$";

  /** The pattern for the AgentHandler. */
  private static final String PATTERN_AGENT = "/agents/\\([^/]+\\)/?$";

  /** The pattern for the AgentControlHandler. */
  private static final String PATTERN_AGENT_CONTROL = "/agents/\\([^/]+\\)/finish/?$";

  /** http client. */
  private HttpClient _httpClient = new HttpClient();

  /** for writing records to JSON without pretty printing. */
  private IpcRecordWriter _ipcRecordWriter = new IpcRecordWriter();

  /**
   * Test for getting the list of available crawler configurations.
   */
  public void testGetAvailableCrawlerConfigurations() throws Exception {
    final RequestHandler service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_CRAWLERS + ")");
    assertTrue(service instanceof CrawlersHandler);
    final CrawlersHandler crawlersHandler = (CrawlersHandler) service;
    final AnyMap result =
      (AnyMap) crawlersHandler.process("GET", crawlersHandler.getRootContextPath() + "/crawlers/", null);
    assertNotNull(result);
    final AnySeq crawlers = result.getSeq("crawlers");
    assertNotNull(crawlers);
    assertEquals(3, crawlers.size());
  }

  /**
   * Test a Crawl run.
   */
  public void testStartCrawl() throws Exception {
    final String jobName = "testStartCrawlWithHandler";
    final String jobRunId = startJob(jobName);
    testStartCrawl(CRAWLER_DATASOURCEID, jobName, false);
    finishJob(jobName, jobRunId);
  }

  /**
   * Test a crawl run twice and expect Exception.
   */
  public void testCrawlerWithExceptions() throws Exception {
    final String jobName = "testStartCrawlWithHandler";
    final String jobRunId = startJob(jobName);
    testStartCrawlWithExceptions(WAIT_CRAWLER_DATASOURCEID, jobName, false);
    finishJob(jobName, jobRunId);
  }

  /**
   * Test a Crawl run with stop.
   */
  public void testStopCrawl() throws Exception {
    final String jobName = "testStopCrawl";
    final String jobRunId = startJob(jobName);
    try {
      // this is correct, true means stopping
      testStartCrawl(WAIT_CRAWLER_DATASOURCEID, jobName, true);
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test for getting the list of available crawler configurations.
   */
  public void testGetAvailableAgentsConfigurations() throws Exception {
    final RequestHandler service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_AGENTS + ")");
    assertTrue(service instanceof AgentsHandler);
    final AgentsHandler agentsHandler = (AgentsHandler) service;
    final AnyMap result =
      (AnyMap) agentsHandler.process("GET", agentsHandler.getRootContextPath() + "/agents/", null);
    assertNotNull(result);
    final AnySeq agents = result.getSeq("agents");
    assertNotNull(agents);
    assertEquals(1, agents.size());
  }

  /**
   * Test an agent run with stop.
   */
  public void testStartAndStopAgent() throws Exception {
    final AgentController agentController = getService(AgentController.class);
    final String jobName = "testStartAndStopAgent";
    final String jobRunId = startJob(jobName);
    try {
      RequestHandler service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_AGENT + ")");
      assertTrue(service instanceof AgentHandler);
      final AgentHandler agentHandler = (AgentHandler) service;
      final String requestUri = agentHandler.getRootContextPath() + "/agents/" + AGENT_DATASOURCEID;
      final Record inputRecord = DataFactory.DEFAULT.createRecord("/agents/([^/]+)/?$");
      inputRecord.getMetadata().put(CrawlerHandler.KEY_JOB_NAME, jobName);
      // start agent.
      agentHandler.process("POST", requestUri, inputRecord);
      Thread.sleep(SLEEP_TIME);

      boolean hasActiveAgents = agentController.hasActiveAgents();
      assertTrue(hasActiveAgents);

      Map<String, AgentState> agentTasksState = agentController.getAgentTasksState();
      assertNotNull(agentTasksState);
      assertEquals(1, agentTasksState.size());
      AgentState agentState = agentTasksState.get(AGENT_DATASOURCEID);
      assertNotNull(agentState);
      assertEquals(AgentThreadState.Running, agentState.getState());
      assertNull(agentState.getLastError());

      Thread.sleep(SLEEP_TIME);
      // check if statistics availablea
      final AnyMap result = (AnyMap) agentHandler.process("GET", requestUri, null);
      assertNotNull(result);
      assertEquals(jobName, result.getStringValue("jobName"));

      service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_AGENT_CONTROL + ")");
      assertTrue(service instanceof AgentControlHandler);
      final AgentControlHandler agentControlHandler = (AgentControlHandler) service;
      // stop agent
      agentControlHandler.process("POST", agentControlHandler.getRootContextPath() + "/agents/"
        + AGENT_DATASOURCEID + "/finish/", inputRecord);
      Thread.sleep(SLEEP_TIME);

      hasActiveAgents = agentController.hasActiveAgents();
      assertFalse(hasActiveAgents);

      agentTasksState = agentController.getAgentTasksState();
      assertNotNull(agentTasksState);
      assertEquals(1, agentTasksState.size());
      agentState = agentTasksState.get(AGENT_DATASOURCEID);
      assertNotNull(agentState);
      assertEquals(AgentThreadState.Stopped, agentState.getState());
      assertNull(agentState.getLastError());
    } finally {
      finishJob(jobName, jobRunId);
    }
  }

  /**
   * Test a Crawl run.
   * 
   * @param dataSourceId
   *          the id of the data source
   * @param jobName
   *          name of the job
   * @param interrupt
   *          the interrupt
   * 
   * @throws Exception
   *           the Exception
   */
  private void testStartCrawl(final String dataSourceId, final String jobName, final boolean interrupt)
    throws Exception {
    final DeltaIndexingManager deltaIndexingManager = getService(DeltaIndexingManager.class);
    deltaIndexingManager.unlockDatasources();

    final CrawlerController crawlerController = getService(CrawlerController.class);
    RequestHandler service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_CRAWLER + ")");
    assertTrue(service instanceof CrawlerHandler);
    final CrawlerHandler crawlerHandler = (CrawlerHandler) service;
    final String requestUri = crawlerHandler.getRootContextPath() + "/crawlers/" + dataSourceId;
    final Record inputRecord = DataFactory.DEFAULT.createRecord("/crawlers/([^/]+)/?$");
    inputRecord.getMetadata().put(CrawlerHandler.KEY_JOB_NAME, jobName);
    // start crawl
    crawlerHandler.process("POST", requestUri, inputRecord);

    final boolean hasActiveCrawls = crawlerController.hasActiveCrawls();
    assertTrue(hasActiveCrawls);

    Map<String, CrawlState> crawlStates = crawlerController.getCrawlerTasksState();
    assertNotNull(crawlStates);
    final CrawlState crawlState = crawlStates.get(dataSourceId);
    assertNotNull(crawlState);
    assertEquals(CrawlThreadState.Running, crawlState.getState());
    assertNull(crawlState.getLastError());

    // check if statistics available
    final AnyMap result = (AnyMap) crawlerHandler.process("GET", requestUri, null);
    assertNotNull(result);
    assertEquals(jobName, result.getStringValue("jobName"));

    if (interrupt) {
      service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_CRAWLER_CONTROL + ")");
      assertTrue(service instanceof CrawlerControlHandler);
      final CrawlerControlHandler crawlerControlHandler = (CrawlerControlHandler) service;
      // stop crawler
      crawlerControlHandler.process("POST", crawlerControlHandler.getRootContextPath() + "/crawlers/"
        + dataSourceId + "/finish/", inputRecord);
    }
    // wait for crawl to stop
    Thread.sleep(SLEEP_TIME);
    while (CrawlThreadState.Running.equals(crawlStates.get(dataSourceId).getState())) {
      Thread.sleep(SLEEP_TIME);
      crawlStates = crawlerController.getCrawlerTasksState();
    }
  }

  /**
   * Test crawl runs with exceptions.
   * 
   * @param dataSourceId
   *          the id of the data source
   * @param jobName
   *          name of the job
   * @param interrupt
   *          the interrupt
   * 
   * @throws Exception
   *           the Exception
   */
  private void testStartCrawlWithExceptions(final String dataSourceId, final String jobName, final boolean interrupt)
    throws Exception {
    final DeltaIndexingManager deltaIndexingManager = getService(DeltaIndexingManager.class);
    deltaIndexingManager.unlockDatasources();

    final RequestHandler service = getService(RequestHandler.class, "(uriPattern=" + PATTERN_CRAWLER + ")");
    assertTrue(service instanceof CrawlerHandler);
    final CrawlerHandler crawlerHandler = (CrawlerHandler) service;
    final String requestUri = crawlerHandler.getRootContextPath() + "/crawlers/" + dataSourceId;
    final Record inputRecord = DataFactory.DEFAULT.createRecord("/crawlers/([^/]+)/?$");
    inputRecord.getMetadata().put(CrawlerHandler.KEY_JOB_NAME, jobName);
    final PostMethod method = new PostMethod("http://localhost:8080" + requestUri);
    final RequestEntity requestEntity =
      new StringRequestEntity(_ipcRecordWriter.writeJsonObject(inputRecord), "application/json", "UTF-8");
    method.setRequestEntity(requestEntity);
    // start crawl
    _httpClient.executeMethod(method);
    // start same crawl again, BadRequest expected
    int httpResponseCode = _httpClient.executeMethod(method);
    assertEquals(HttpStatus.BAD_REQUEST, httpResponseCode);
    // get information with non existing data source id, NotFound expected
    final GetMethod getMethod = new GetMethod("http://localhost:8080" + requestUri + "NONEXISTING");
    httpResponseCode = _httpClient.executeMethod(getMethod);
    assertEquals(HttpStatus.NOT_FOUND, httpResponseCode);
  }
}
